Unit WaveUnit;
{ Copyright 1999 Sami Silaste          }
{ <samis@aethos.co.uk>
  <sami@semaphore.demon.co.uk>         }
{ Wave format info from WAVIO.pas of
  Don Cross <dcross@intersrv.com>      }

interface

uses utils,filesys;

Function GetWaveInfo (Fina:string; var Format:word; var Bits:word; var Rate:longint; var Channels:word):boolean;
Function CheckWave(wname:string):longint;
function readbyte:byte;

const
    stereo:boolean=false;
    bits16:boolean=false;
    lobit_def:byte=10;
    hibit_def:byte=14;

implementation

const
  MaxWaveChannels = 4;

type
  WaveFileSample = array [0 .. MaxWaveChannels-1] of integer;

  RiffChunkHeader = record
    ckID:     longint;    { four-character chunk ID }
    ckSize:   longint;    { length of data in chunk }
  end;

  WaveFormat_ChunkData = record
    wFormatTag:         word;
    nChannels:          word;
    nSamplesPerSec:     longint;
    nAvgBytesPerSec:    longint;
    nBlockAlign:        word;
    nBitsPerSample:     word;
  end;

  WaveFormat_Chunk = record
    header:   RiffChunkHeader;
    data:     WaveFormat_ChunkData;
  end;

var
  riffHeader:RiffChunkHeader;
  WaveFormat:WaveFormat_Chunk;
  pcmData:WaveFormat_ChunkData;
  pcmDataOffset,numSamples:longint;

function FourCC ( ChunkName: string ): longint;
var
  retbuf: longint;
  i, shift: integer;
  c: char;

begin
  retbuf := 0;
  for i := 4 downto 1 do begin
    retbuf := retbuf SHL 8;
    if i <= Length(ChunkName) then
      c := ChunkName[i]
    else
      c := ' ';
      retbuf := retbuf OR longint(c);
    end;
    FourCC := retbuf;
end;

function BitsPerSample: word;
begin
  BitsPerSample := waveFormat.data.nBitsPerSample;
end;

function NumChannels: word;
begin
  NumChannels := waveFormat.data.nChannels;
end;

function SamplingRate: longint;
begin
  SamplingRate := waveFormat.data.nSamplesPerSec;
end;

function SampleFormat: word;
begin
  SampleFormat := waveFormat.data.wFormatTag;
end;

procedure OpenWAV(filename: string; var success: boolean);
var
    numRead: word;
    signature: array [0..3] of char;

begin
  {$I-}
  reset (sfile, 1);
  {$I+}
  success := (IOResult = 0);
  if not success then exit;
  BlockRead ( sfile, riffHeader, sizeof(riffHeader), numRead );
  if (numRead <> sizeof(riffHeader)) or
     (riffHeader.ckID <> FourCC('RIFF')) then
  begin
    success := FALSE;
    exit;
  end;

  BlockRead ( sfile, signature, 4, numRead );
  if (numRead <> 4) or
     (signature[0] <> 'W') or
     (signature[1] <> 'A') or
     (signature[2] <> 'V') or
     (signature[3] <> 'E') then
  begin
    success := FALSE;
    exit;
  end;

  BlockRead ( sfile, waveFormat, sizeof(waveFormat), numRead );
  if (numRead <> sizeof(waveFormat)) or
     (waveFormat.header.ckID <> FourCC('fmt')) or
     (waveFormat.data.nChannels < 1) or
     (waveFormat.data.nChannels > MaxWaveChannels) then
  begin
    success := FALSE;
    exit;
  end;

  pcmDataOffset := FilePos(sfile);
  BlockRead ( sfile, pcmData, sizeof(pcmData), numRead );
  if numRead <> sizeof(pcmData) then
  begin
    success := FALSE;
    exit;
  end;

  numSamples := (FileSize(sfile) - FilePos(sfile)) DIV (NumChannels * BitsPerSample DIV 8);
end;

Procedure WaveInit;
begin
  fileOpen := FALSE;
  riffHeader.ckID := FourCC('RIFF');
  riffHeader.ckSize := 4 + sizeof(waveFormat) + sizeof(riffHeader);
  pcmDataOffset := 0;
  numSamples := 0;
  riffheader.ckID := FourCC('data');
  riffheader.ckSize := 0;
  waveFormat.header.ckID := FourCC('fmt');
  waveFormat.header.ckSize := sizeof(waveFormat.data);
end;

Function GetWaveInfo (Fina:string; var Format:word; var Bits:word; var Rate:longint; var Channels:word):boolean;
var opensuccess:boolean;
begin
  WaveInit;
  OpenWAV(fina,opensuccess);
  if opensuccess then
  begin
    format := SampleFormat;
    bits := BitsPerSample;
    rate := SamplingRate;
    channels := NumChannels;
  end;
  GetWaveInfo := opensuccess;
end;

Function CheckWave(wname:string):longint;
{ Determine WAV format }
const chinfo : array[1..2] of string = ('Mono','Stereo');
var f,b,c:word;
    r:longint;
    waveOK:boolean;
begin
  waveOK := true;
  if GetWaveInfo (wname , f, b, r, c) then
  begin
    if f=1 then
      Xwriteln('Format:PCM')
    else
      Xwriteln('ERROR: Non-PCM Wave format.');
    if not (b in [8,16]) then Xwriteln('ERROR: Not 8 or 16 bit sample');
    Xwriteln('Bits:'+Wordval(b));
    if r < 8192 then Xwriteln('ERROR: Rate is too low');
    Xwriteln('Rate:'+WordVal(r));
    if c in [1,2] then
      Xwriteln('Channels:'+WordVal(c)+' ('+chinfo[c]+')')
    else
      Xwriteln('ERROR: Too many channels:'+WordVal(c));
    if (f<>1) or (not(b in[8,16])) or (not(c in [1,2])) then waveOK := false;
  end
  else
  begin
    Xwriteln('ERROR: file not found.');
    waveOK := false;
  end;
  if c=2 then stereo := true;
  if b=16 then bits16 := true;
  if not waveOK then CheckWave := 0 else
  begin CheckWave := r; lobit_def := r div 4410; hibit_def := r div 3150; end;
end;

function readbytes:byte;
{ Reduces a 16 bit sample to 8 bits and returns one logical byte. }
var eightbits1,eightbits2:byte;tmp:word;
    tmpint: integer absolute tmp;
begin
  if bits16 then
  begin
    eightbits1 := readfilebyte(sfile);
    eightbits2 := 0;
    if not endoffile then eightbits2 := readfilebyte(sfile);
    tmp := (eightbits1) or (eightbits2 shl 8);
    readbytes := byte((tmpint+$8000) shr 8);
  end
  else
    readbytes := readfilebyte(sfile);
end;

function readbyte:byte;
{ Combines stereo channels to one mono channel and returns one logical byte. }
var lchannel,rchannel,tbyte:byte;
begin
  if stereo then
  begin
    lchannel := readbytes;
    rchannel := 0;
    if not endoffile then rchannel := readbytes;
    case channel of
    'L': tbyte := lchannel;
    'R': tbyte := rchannel;
    'B': tbyte := (lchannel + rchannel) div 2; { average stereo channels }
    end;
  end
  else
    tbyte := readbytes;
  readbyte := tbyte xor inverse;
end;


end.