//
// NAME:
//   im_pcx.h
// TITLE:
//   Image: PCX Object.
// FUNCTION:
//   See header.
//
// AUTHOR:
//   Kent J. Quirk, from a Dr. Dobbs tutorial.
//   C++ Conversion for Image Library by Brendan Jones, Kestrel Defence.
//
// RIGHTS:
//   (c) Copyright Kent J. Quirk, 1994.  All Rights Reserved.
//   Conversion (c) Copyright Brendan Jones, 1994.  All Rights Reserved.
// SECURITY:
//   Unclassified.  For Public Release.
// DATE:
//   November 30, 1994.
//
// MODIFICATIONS:
//   NAME  MOD  DATE       DESCRIPTION
//
//
//
#include <stdio.h>
#include <stdlib.h>
#ifdef MACOS
#include<malloc/malloc.h>
#else
#include <malloc.h>
#endif
#include <memory.h>
#include "image/im_pcx.h"



im_PCX::im_PCX(char *filename)
{
  if ((pcx_f = fopen(filename, "rb")) == NULL)
    abend(WHERE0, "open fails");

  pcx_read_header(&header, pcx_f);

  mcheck(line = pcx_alloc_line(&header));

  // How many lines are there in this file?
  lines = rows();

  // Load the palette
  //
  if (header.version == 5 and header.bpp == 8)
    {
    long linepos = ftell(pcx_f);
    byte magic;

    if (fseek(pcx_f, -769, 2) != 0)
      abend(WHERE0, "Seek fails");

    if (fread(&magic, 1, 1, pcx_f) != 1)
      abend(WHERE0, "Bad read");

    if (magic != 12)
      abend(WHERE0, "Bad PCX magic number");

    static_palette = false;
    mcheck(palette = new byte [768]);

    if (fread(palette, sizeof(byte), 768, pcx_f) != 768)
      abend(WHERE0, "Read fails");

    if (fseek(pcx_f, linepos, 0) != 0)
      abend(WHERE0, "Seek fails");
    }
  else
    {
    static_palette= true;
    palette = header.palette;
    }
}




im_PCX::~im_PCX(void)
{
  if (line != NULL)
    delete[] line;

  if (pcx_f != NULL)
    fclose(pcx_f);

  if (not static_palette)
    delete[] palette;
}




// FUNCTION Image PCX Next Line.
//
// Read the next line in the image into the buffer.
//
// OUT:
//   line	The line is stored here
//
// RESULT:
//   True if we've hit EOF.
//
//
boolean im_PCX::next_line(void)
{
  if (lines == 0)
    {
    if (line != NULL)
      {
      delete[] line;
      line = NULL;
      }
    return true;
    }
  else
    {
    pcx_next_line(&header, line, pcx_f);

    if (--lines == 0)
      {
      fclose(pcx_f);
      pcx_f = NULL;
      }

    return false;
    }
}




ostream& operator << (ostream& str, im_PCX& P)
{
    char *t;
    if (P.header.pcx_id != 0x0A)
    {
	str << "Not a PCX file.\n";
	return str;
    }
    switch (P.header.version) {
    case 0: t="2.5"; break;
    case 2: t="2.8 with palette info."; break;
    case 3: t="2.8 without palette info."; break;
    case 5: t="3.0"; break;
    default: t="unknown."; break;
    }
    str << "PCX Version: " << t << nl;
    str << "Compression: "
	<< (P.header.encoding==1 ? "Run length" : "Unknown") << nl;
    str << (ushort)P.header.bpp << "bits per pixel\n";
    str << "Image from: "
	<<  P.header.upleftx  << "," << P.header.uplefty << " to "
	<<  P.header.lorightx << "," << P.header.lorighty << nl;
    str << "Created on a device with "
	<< P.header.display_xres << " by " << P.header.display_yres
	<< " dpi resolution.\n";
    str << "The image contains " << (ushort)P.header.nplanes
	<< " image planes.\n";
    str << "There are " << P.header.bytesperline
	<< " bytes per line of data after decompression.\n",
    str << "There are "
	<< ((long)P.header.lorighty - (long)P.header.uplefty + 1) *
	    (long)P.header.bytesperline 
	<< " total bytes in the image.\n";

    return str;
}






// MODULE Kent J. Quirk's Original Routines.
//
//


/**** p c x _ r e a d _ h e a d e r ****
    Abstract:   Reads the header of a PCX file.
    Parameters: A data storage area for the header, an open file.
    Returns:    The pointer to the data storage area passed, or NULL if error.
    Comments:   The file should be opened in binary mode.
****************************/
im_PCX_header *im_PCX::pcx_read_header(im_PCX_header *hdr, FILE *f)
{
    fseek(f, 0L, SEEK_SET);     /* header is at top of file */
    if (fread(hdr, 1, sizeof(im_PCX_header), f) != sizeof(im_PCX_header))
	return(NULL);
    else
	return(hdr);
}

/**** p c x _ p r i n t _ h e a d e r ****
    Abstract:   Printf's a PCX file header data to a given file.
    Parameters: The PCX file header, the file to write the data to.
    Returns:    Nothing
****************************/
void im_PCX::pcx_print_header(im_PCX_header *hdr, FILE *f)
{
  char *t;

    if (hdr->pcx_id != 0x0A)
    {
	fprintf(f, "Not a PCX file.\n");
	return;
    }
    switch (hdr->version) {
    case 0: t="2.5"; break;
    case 2: t="2.8 with palette info."; break;
    case 3: t="2.8 without palette info."; break;
    case 5: t="3.0"; break;
    default: t="unknown."; break;
    }
    fprintf(f, "PCX Version %s\n", t);
    fprintf(f, "Compression: %s\n",
		hdr->encoding==1 ? "Run length" : "Unknown");
    fprintf(f, "%d bits per pixel\n", hdr->bpp);
    fprintf(f, "Image from (%d, %d) to (%d, %d) pixels.\n",
	    hdr->upleftx, hdr->uplefty, hdr->lorightx, hdr->lorighty);
    fprintf(f, "Created on a device with %d x %d dpi resolution.\n",
	    hdr->display_xres, hdr->display_yres);
    fprintf(f, "The image contains %d image planes.\n", hdr->nplanes);     fprintf(f, "There are %d bytes per line of data after decompression.\n",
		hdr->bytesperline);
    fprintf(f, "There are %ld total bytes in the image.\n",
		((long)hdr->lorighty - (long)hdr->uplefty + 1) *
		 (long)hdr->bytesperline);
    return;
}

/**** p c x _ a l l o c _ l i n e ****
    Abstract:   Allocates enough space to store a complete scan line from the
		current PCX file.
    Parameters: The header pointer.
    Returns:    A pointer to a "big enough" block of allocated space, or
		null if not enough space or an error occurred.
****************************/
byte *im_PCX::pcx_alloc_line(im_PCX_header *hdr)
{
  // 30Nov94 bj: changed to use new, thus making it independant
  // of the memory model.  We also make sure the image isn't null.
  //
  // was: return  ((byte *)calloc(hdr->nplanes, hdr->bytesperline));
  //
  if (hdr->nplanes * hdr->bytesperline > 0)
    return new byte [hdr->nplanes * hdr->bytesperline];
  else
    return NULL;
}

/**** p c x _ n e x t _ l i n e ****
    Abstract:   Reads the next line from the PCX file.
    Parameters: Pointer to the header, a pointer to the line area (or NULL
		for automatic allocation), and the open data file.
    Returns:    A pointer to the line area, or NULL if there was a problem.
    Comments:   To read the first line, call pcx_read_header() first.
		This sets the file position to the beginning of the data.
****************************/
byte *im_PCX::pcx_next_line(im_PCX_header *hdr, byte *line, FILE *f)
{
    int c, len;
    byte *dp;
    word linesize = hdr->nplanes * hdr->bytesperline;
    word i;
    if (line == NULL)
	if ((line = pcx_alloc_line(hdr)) == NULL)
	    return(line);
    dp = line;                  /* point to data */
    for (i=0; i<linesize; )
    {
	if ((c = fgetc(f)) == EOF)
	    return(NULL);
	if ((c & PCX_COMPRESSED) == PCX_COMPRESSED)
	{
	    len = (c & PCX_MASK);
	    if ((c = fgetc(f)) == EOF)
		return(NULL);
	    memset(dp, (byte)c, len);
	    dp += len;
	    i += len;
	}
	else
	{
	    *dp++ = (byte)c;
	    i++;
	}
    }
    return(line);
}
