//
// NAME:
//   common.h 2.0 (Quetzalcoatl Version)
// TITLE:
//   Common constants, type definitions and macros.  (COMMON2)
// FUNCTION:
//   In 1992 The Common Library was created.  It includes not just this
//   header, but many reusable objects offering services ranging from
//   graphics to lexical parsing.
//
//   This header file includes frequently used header files, and declares
//   frequently used constants and macros.  Apart from abend, other 
//   Common libraries should be explicitly included after this header
//   file is included.  Abend is used so frequently it is included by
//   default.
//
// AUTHOR:
//   Brendan Jones. (Contact through www.kdef.com/geek/vic)
// RIGHTS:
//   Copyright Brendan Jones, 1992-2006.
// SECURITY:
//   Unclassified.  
// LEGAL NOTICE:
//   See legal.txt before viewing, modifying or using this software.
// CONTACT:
//   Web:	http://www.kdef.com/geek/vic
//   Email:	See www.kdef.com/geek/vic
// DATE:
//   April 10, 1992.
// RIGHTS:
//  This file is part of The Quetzalcoatl Compiler.
//  
//  The Quetzalcoatl Compiler is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//  
//  The Quetzalcoatl Compiler is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//  
//  You should have received a copy of the GNU General Public License
//  along with The Quetzalcoatl Compiler; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
//
// MODIFICATIONS:
//   NAME  MODNO  DATE     DESCRIPTION
//   bj    1      29jan95  Version 2.0.  Added Unix compatibility.
//   Moved a lot of unused code into a section for backward compatibility.
//   This will only be included if COMMON2 is not defined.
//   bj    2      16apr95  Added bread, bread1 and bwrite macros.
//   bj    3       2oct95  Add bwrite1; put bwrite in curly brackets so it parses as a
//   single statement.  Added KESTREL_COPYRIGHT string and size macros.
//   bj    4      15mar96  Added round() macro.
//   bj    5      23feb97  Made COMMON2 the default.   Added ANSI MSVC++ aliases.
//			   Moved iabs() into backwards compatibility section;
//			   replaced with absolute().  Added rareif().
//			   Added insist() function; like assert(), but remains
//			   in effect even if compiled with NDEBUG.
//   bj    6      28mar98  Set <max_filename> to 260 for Windows,
//                         512 for Posix.
//   bj    7       1jan99  Add mathematical constants, remove real datatype.
//
//
#ifndef COMMON_H
#define COMMON_H
//
// Determine Common Release.
//
// Common 1.0 compliant programs must declare COMMON1 before calling us.
// Anything else must be a Common 2.0 compliant program, so declare
// COMMON1 accordingly.
//
//
#ifndef COMMON1
#define COMMON2
#endif


// Standard Header Files
//
#include <stdlib.h>
#ifdef __MSDOS__
#include <conio.h>
#include <alloc.h>
#elif	defined(__unix__)
#include <unistd.h>
#endif
#include <math.h>
#include <stdio.h>
#include "implemen.h"

// FIX: A couple of years ago some self-important technocrats with too much 
//      time on their hands decided to rewrite the C++ Iostream classes, 
//	breaking nearly every C++ program in existance for no good reason. Really. 
//	There was no apparent change to program behaviour, and the changes
//	didn't give us anything of any practical benefit. All they did was force 
//	everyone to change rearrange a few arbitrary parameters and change our 
//	includes to drop the ".h".  Versions of MSVC (2003+) insist on the changes, 
//	so we must do their idiot include. I don't know exactly who these
//      technocrats were, but if anybody does can you please make sure they're
//      given real jobs so they don't waste our time with more pointless reorgs?
//	[bj 08oct2006]
//
#ifdef COMMON_MSVC
#include <fstream>
#include <iomanip>
#include <ios>
#include <iosfwd>
#include <iostream>
#include <istream>
#include <ostream>
#include <sstream>
#include <streambuf>
#include <strstream>

using std::endl;
using std::flush;
using std::cout;
using std::cerr;
using std::cin;
using std::istream;
using std::ostream;
using std::fstream;
using std::ifstream;
using std::ofstream;
using std::strstream;
using std::istrstream;
using std::ostrstream;
using std::filebuf;
using std::hex;
using std::dec;
using std::oct;
using std::endl;
#else

// Or you could just do this... :-| [bj 08oct2006]
//
#include <iostream.h>
#endif

#include <string.h>
#include <assert.h>


// Copyright Statement.
//
// Many data files need a copyright statement.  For convenience this statement
// and its size are stored as these macros.  A carriage return, linefeed and
// control-z are included so the statement prints cleanly under MS-DOS.
// Additional spaces are included to allow dates to be added or revised.
//
#define KESTREL_COPYRIGHT       "Copyright Brendan Jones, 2006.  All Rights Reserved.\000\015\012\032                                "
#define KESTREL_COPYRIGHT_SIZE  90


// Constants
//
#define UINVALID  (65530u)   /* An invalid value in a ushort */
#define INVALID   (-32760)   /* An invalid value to a short  */


// Types
//
typedef unsigned char           byte;           // 0..255
typedef unsigned short          word;           // 0..65535
typedef unsigned short int      ushort;         // 0..65535
typedef unsigned long           doubleword;     // 0..4294967296
typedef unsigned long           dword;          // 0..4294967296
typedef unsigned long int       ulong;          // 0..4294967296
typedef unsigned int            uint;           // 0..65535 on 16 bits
						// 0..4294967297 on 32 bits


// Type: Boolean
//
typedef unsigned char   boolean;
#ifndef true
#define true		1
#endif
#ifndef false
#define false		0
#endif


// CONSTANT Largest Filename.
//
// The largest filename we can reasonably expect.
//
#ifdef COMMON_DOS
#define max_filename            128
#elif defined(COMMON_POSIX)
#define max_filename            512
#elif defined(COMMON_MSW)
#define max_filename            260
#else
#define max_filename            256
#endif




// GROUP Macros.
//


// A newline character
#define nl      '\n'



// GROUP Control Macros.
//


// A loop construct
#define loop            for(;;)


// This macro simulates the REXX select statement.  It resembles a switch
// in C, but you can use expressions instead of just constants.
//
// In Common 2.0 "select" was replaced with "Select" to prevent a clash
// with the POSIX operating system call of the same name.
//
//
#define Select(X)       {if (X) {
#define when(X)         } else if (X) {
#define otherwise       } else {
#define endseltrap      } else {abend(WHERE, 0, "Select Trap");}}
#define endsel          }}
#define endselect       }}


// Swap to variables using a temporary
#define swap(X, Y, T)   {T=X; X=Y; Y=T;}

#ifndef NEWCPP
// Boolean operators we can actually understand
#define and             &&
#define bitand          &
#define or              ||
#define bitor           |
#define xor             ^
#define bitxor          ^
#define not             !       /* Actually, I don't mind this one */
#define bitnot          ~
#endif


// Find the maximum or maximum values from a set of two numbers.
#ifndef	min
#define min(X, Y)       ((X) < (Y) ? (X) : (Y))
#endif

#ifndef	max
#define max(X, Y)       ((X) > (Y) ? (X) : (Y))
#endif

// Absolute value of two integers
#define absolute(X)	((X) < 0 ? -(X) : (X))

// The square of two numbers
#define square(X)       ((X)*(X))
#define squared(X)      ((X)*(X))

// Round a floating point number to the nearest integer.
#define rounded(X)        ((X)+0.5)


// Macro to convert between endian formats.
//
#define ltohes(X)       ((((unsigned)X)<<8) bitor (((unsigned)X)>>8))
#define htoles(X)       ((((unsigned)X)<<8) bitor (((unsigned)X)>>8))


// True iff two floating point numbers are roughly equal.
#define req(X,Y)	(fabs((X)  - (Y)) <= EPSILON_FLOAT)

// True iff two floating point numbers are not even roughly equal.
#define rneq(X,Y)       (fabs((X) - (Y))  >  EPSILON_FLOAT)

// True iff a number is roughly equal to zero.
#define rzero(X)        (fabs(X) <= EPSILON_FLOAT)


// Compare two strings (unlike strcmp these are easy to understand)
#define equal(X, Y)     (strcmp((X), (Y)) == 0)
#define notequal(X, Y)  (strcmp((X), (Y)) != 0)





// GROUP File Macros.
//

// Modes for opening files.
//
#define READ_TEXT               "r"
#define READ_BINARY             "rb"
#define WRITE_TEXT              "w"
#define WRITE_BINARY            "wb"
#define APPEND_TEXT             "w+"
#define APPEND_BINARY           "wb+"


// Open a file, abending if the open fails.
//
#define file_open(File, Filename, Mode)				\
	{                                                       \
	if ((File=fopen(Filename, Mode)) == NULL)		\
	  abend(WHERE0, "Open fails", 0, Filename);		\
	}


#define	file_close(File)	{fclose(File);}


// Reading and writing to binary files is painful as every operation needs to be
// checked.  The following macros simplify the parameter specfication by deducing
// most of the parameters automatically.  It also inserts a check after every
// operation.
//

// Write <count> objects to a binary file.
//
#define bwrite(ptr, count, file)                                \
	{                                                       \
	if (fwrite(ptr, sizeof(*ptr), count, file) != count)    \
	  abend(WHERE0, "Write failed");                        \
	}

// Write a single object to a binary file.
//
#define bwrite1(ptr, file)                                      \
	{                                                       \
	if (fwrite(ptr, sizeof(*ptr), 1, file) != 1)            \
	  abend(WHERE0, "Write failed");                        \
	}

// Create and read <count> objects from a binary file.
//
#define breadc(ptr, type, count, file)                          \
	{                                                       \
	mcheck(ptr = new type [count]);                         \
								\
	if (fread(ptr, sizeof(*ptr), count, file) != count)     \
	  abend(WHERE0, "Read failed");                         \
	}

// Create and read a single object from a binary file.
//
#define breadc1(ptr, type, file)                                \
	{                                                       \
	mcheck(ptr = new type);                                 \
								\
	if (fread(ptr, sizeof(*ptr), 1, file) != 1)             \
	  abend(WHERE0, "Read failed");                         \
	}

// Read an object from a binary file.
//
#define bread1(ptr, file)                                       \
	{                                                       \
	if (fread(ptr, sizeof(*ptr), 1, file) != 1)             \
	  abend(WHERE0, "Read failed");                         \
	}

// Read an <count> objects from a binary file.
//
#define bread(ptr, count, file)                                 \
	{                                                       \
	if (fread(ptr, sizeof(*ptr), count, file) != count)     \
	  abend(WHERE0, "Read failed");                         \
	}



// GROUP Compiler hints.
//

// Based on Lint's /*NOTREACHED*/ comment.
//
// We do an exit here since some compilers insist every function path returns a
// value, even if that function actually abends.  Unfortunately for some compilers
// even this isn't enough.
//
#define NOTREACHED      exit(8);

// Make the compiler think a variable is in fact used
#define use(X)          X = (X)




// GROUP Integrity.
//

// Make sure a pointer is not NULL
//
// This macro checks for a null pointer in a memory reference.
#define mcheck(X)       {if ((X) == NULL) \
			abend(WHERE0, "Out of Memory");}
//
// This macro makes sure a pointer is not null; eg. after an fopen.
#define nncheck(X)      {if ((X) == NULL) \
			abend(WHERE0, "Operation failed; got a null pointer.");}




// Microsoft Visual C++.
//
// Some of Microsoft Visual C++'s ANSI function names have a '_' character in
// front of them.  These aliases let us use the standard ANSI function names.
//
//
#ifdef _MSC_VER
#define	access(Filename, Mode)	_access(Filename, Mode)
#endif

// swap endianness for Suns
// HFD
#ifdef FIXEND
#define wordswap(theword) {			\
		unsigned int tempword;		\
		tempword = (theword >> 8); 	\
		theword = tempword | (theword << 8); }

#define longswap(thelong) {					\
		unsigned long templong;				\
		templong = ((thelong >> 24) & 0x000000ff);	\
		templong |= ((thelong >> 8) & 0x0000ff00);	\
		templong |= ((thelong << 8) & 0x00ff0000);	\
		thelong = templong | ((thelong << 24) & 0xff000000); }
#endif


// Common library header files.
//
#include "abend.h"
#endif
