//
// NAME:
//   crc.cpp
// TITLE:
//   Common: Cyclic Redundancy Check.
// FUNCTION:
//   See header.
//
// AUTHOR:
//   Anonymous, from CRC.ZIP on the Walnut Creek 1992 Source CD-ROM.
//   Brendan Jones. (Contact through www.kdef.com/geek/vic)
// RIGHTS:
//   Conversion (c) Copyright Brendan Jones, 1995.  All Rights Reserved.
// SECURITY:
//   Unclassified.  
// LEGAL NOTICE:
//   See legal.txt before viewing, modifying or using this software.
// CONTACT:
//   Web:	http://www.kdef.com/geek/vic
//   Email:	See www.kdef.com/geek/vic
// DATE:
//   August 20, 1995.
// RIGHTS:
//  This file is part of The Quetzalcoatl Compiler.
//  
//  The Quetzalcoatl Compiler is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//  
//  The Quetzalcoatl Compiler is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//  
//  You should have received a copy of the GNU General Public License
//  along with The Quetzalcoatl Compiler; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
//
// MODIFICATIONS:
//   NAME  MOD  DATE       DESCRIPTION
//
//
#include "crc.h"


// PROCEDURE CRC16 open.
//
// This function clears the CRC to zero. It should be called prior to
// the start of the processing of the stream.
//
//
void  CRC16::open(void)
{
  crc = 0;
}




// PROCEDURE CRC16 Process.
//
// This function must be called once for each character which is
// to be included in the CRC for messages to be transmitted.
// This function is called once for each character which is included
// in the CRC of a received message, AND once for each of the two CRC
// characters at the end of the received message. If the resulting
// CRC is zero, then the message has been correctly received.
//
// IN:
//   crc_char		The byte of data being processed.
//
//
void CRC16::process(byte crc_char)
{
  long x;
  short i;

  // "x" will contain the character to be processed in bits 0-7 and the CRC
  // in bits 8-23. Bit 24 will be used to test for overflow, and then cleared
  // to prevent the sign bit of "x" from being set to 1. Bits 25-31 are not
  // used. ("x" is treated as though it is a 32 bit register).

  x = ((long)crc << 8) + crc_char;    	// Get the CRC and the character


  // Repeat the following loop 8 times (for the 8 bits of the character).
  for (i=0; i<8; i++)
    {

    // Shift the high-order bit of the character into the low-order bit of
    // the CRC, and shift the high-order bit of the CRC into bit 24.
    //
    x = x << 1;                        // Shift "x" left one bit

    // Test to see if the old high-order bit of the CRC was a 1.
    //
    if (x & 0x01000000)                // Test bit 24 of "x"

  // If the old high-order bit of the CRC was a 1, exclusive-or it with a one
  // to set it to 0, and exclusive-or the CRC with hex 1021 to produce the
  // CCITT-recommended CRC generator of: X**16 + X**12 + X**5 + 1. To produce
  // the CRC generator of: X**16 + X**15 + X**2 + 1, change the constant from
  // 0x01102100 to 0x01800500. This will exclusive-or the CRC with hex 8005
  // and produce the same CRC that IBM uses for their synchronous transmission
  // protocols.

      x = x ^ 0x01102100;   		// Exclusive-or "x" with a...
					// ...constant of hex 01102100
    }


  // Return the CRC as the 16 low-order bits of this function's value.
  crc = (((x & 0x00ffff00) >> 8)); // AND off the unneeded bits and...
				   // ...shift the result 8 bits to the right
}




// PROCEDURE CRC16 close.
//
// Close the byte data stream, returning the calculated crc.
//
// RESULT:
//   crc	The checksum generated for the stream.
//
//
crc16 CRC16::close(void)
{
  return crc;
}
