//
// NAME:
//   flex.cpp
// TITLE:
//   Lexical Parser.
// FUNCTION:
//   See header.
//
// AUTHOR:
//   Brendan Jones.
// RIGHTS:
//   (c) Copyright Brendan Jones, 1995-1999.  All Rights Reserved.
// LEGAL NOTICE:
//   See legal.txt before viewing, modifying or using this software.
// CONTACT:
//   Web:	http://www.kdef.com/geek/vic
//   Email:	quetzalcoatl@kdef.com
// DATE:
//   December 31, 1995.
// RIGHTS:
//  This file is part of The Quetzalcoatl Compiler.
//  
//  The Quetzalcoatl Compiler is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//  
//  The Quetzalcoatl Compiler is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//  
//  You should have received a copy of the GNU General Public License
//  along with The Quetzalcoatl Compiler; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
//
// MODIFICATIONS:
//   [see header]
//   NAME  MOD  DATE     DESCRIPTION
//   bj    1    10apr96  Added hex.
//   bj    2    17apr96  Added support for escaped characters inside
//			 strings, including multiline strings.  Added
//			 a string length test to make sure it doesn't
//			 exceed the buffer length.
//   bj    3     6aug97  Add parse_warning(), string_convert() support
//			 and don't accept strings for check().
//   bj    4  21sep1998  Added ++, --.
//   bj    5  15apr99    Added set_comment_char().
//   bj    6  09oct2006  Rolled back to the original flex.cpp code;
//			 see solaris/does-not-work-under-MSVC
//			 for Harry's GCC port. That version had an
//			 infinite loop under Windows.
//
//
#include <ctype.h>
#include "flex.h"




// CONSTRUCTOR Flex.
//
//
Flex::Flex(long Max_token_size)
{
  buffer 	    = NULL;	// We are not parsing a string.
  file	 	    = NULL;	// We are not parsing a file.
  string_conversion = true;	// We are converting slash sequences.
  string_enclosure  = true;
  appl_comment_char =    0;

  mcheck(lookahead      = new char [flex_token_size = Max_token_size]);
  mcheck(lookahead_copy = new char [flex_token_size]);
}




// DECONSTRUCTOR Flex.
//
//
Flex::~Flex(void)
{
  close();

  if (lookahead)
    delete[] lookahead;

  if (lookahead_copy)
    delete[] lookahead_copy;
}




// PROCEDURE Flex Init.
//
// Initialise variables common to both file and string parsing.
//
//
void Flex::init(void)
{
  token_lines  	=
  lines        	= 0;		// Reset line counter.
  token_columns =
  columns      	= 0;		// Reset column counter.
  ch_available 	= false;	// No character is currently loaded in <ch>.
  exhausted    	= false;	// We haven't hit the end of the file yet.
  abort	       	= true;		// Errors are fatal.
  error	      	=
  warning      	= 0;		// No warning/error has yet been encountered.
  space_before  = false;

  read_next();			// Read the first token into <lookahead>.
}




// PROCEDURE Flex Open File.
//
// Open the file <Filename> for parsing.
//
// IN:
//   Filename		The file being parsed.
//   Syntax		What syntax are we passing?
//
// ABENDS:
//   If the file cannot be opened for reading.
//
//
void Flex::open_file(const char *Filename, flex_syntax Syntax)
{
  if ((file = fopen(Filename, "r")) == NULL)  // Read the file in text mode.
    abend(WHERE0, "Could not open file", 0, Filename);

  strcpy(source, Filename);		     // Remember the filename.
  syntax = Syntax;			     // Remember the syntax.


  // Initialise the variables and read the first token.
  //
  init();
}




// PROCEDURE Flex Open String.
//
// Prepare to parse a null-terminated string.
//
// IN:
//   Data               The null-terminated string
//   Source		Optional description for error reporting purposes.
//			eg. "The proposed rule for salary calculation."
//			Set NULL if not used.
//   Syntax		What syntax are we passing?
//
// NOTE:
//  The string must not be deleted until <Flex> has finished parsing it.
//
//
void Flex::open_string(char *Data, char *Source, flex_syntax Syntax)
{
  buffer = Data;		// <buffer> points to the string.
  index  =    0;		// We are starting with the 0th character.
  syntax = Syntax;		// Remember the syntax.

  // Remember the source.
  //
  if (Source)
    strcpy(source, Source);
  else
    source[0] = 0;


  // Initialise the variables and read the first token.
  //
  init();
}




// PROCEDURE Flex Close.
//
// Close any current parsing operation.
//
//
void Flex::close(void)
{
  if (file)		// If we were parsing a file...
    {
    fclose(file);	// Close it.
    file = NULL;	// Set its pointer to NULL so we know its inactive.
    }

  buffer = NULL;	// We are no longer using the buffer.
}




// PROTECTED FUNCTION Flex Next.
//
// Return the next unprocessed character from the input stream.
//
// By unprocessed we mean that if a character returned by <next()>
// isn't used the caller may elect not to process it by calling
// <reject_next>.  The rejected character will then be held over
// until the next call to <next>.
//
// RESULT:
//   The next unprocessed character from the input stream.
//   0 if the input stream is exhausted.
//
//
char Flex::next(void)
{
  // If no character is currently available in <ch> we must get
  // the next one directly from the input stream.
  //
  if (not ch_available)	// If no character is already available...
    {
    if (exhausted)	// If we already hit the end of the file...
      ch = 0;		// Then set <ch> to 0, meaning exhausted.
    else
      {
      if (buffer)	      // If we're reading from the <buffer> string
	{
	ch = buffer[index++]; // Advance the index and get the next character.

	if (ch == 0)          // If this is the terminating null...
	  exhausted = true;   // Then remember we've hit the end of the file.
	}
      else if (file)	      // If we're reading from a <file>...
	{
	// Get the next character.  If this is the end of the file then
	// set <ch> to 0 (signifying eof) and exhausted to true.
	//
	if ((ch = fgetc(file)) == EOF)
	  {
	  ch = 0;
	  exhausted = true;
	  }
	}
      else
	// They haven't opened a file or data string source, so abend.
	//
	{
	abend(WHERE0, "No source opened");
        return  0;      // NOTREACHED
        }


      // Increment our recorded location within the source.
      //
      if (ch == '\n')	// If this is a linefeed...
	{
	lines++;	// Then we are moving to the next line.
	columns = 0;	// Reset the column counter.
	}
      else
	columns++;	// Increment the column counter.
      }
    }


  // We're returning the character <ch>.  We set <ch_available> to false
  // to signify that <ch> is now considered empty.  However if the caller
  // decides they don't want <ch> they can call <reject_next()> which
  // sets <ch_available> to true.  The next time <next()> is called the
  // character will be taken directly from <ch> rather than from the
  // input stream.  For now though we assume that <ch> is no longer
  // available.
  //
  ch_available = false;


  // Return the next character.
  //
  return ch;
}




// PROCEDURE Flex Read Next.
//
// Read the next token from the input stream into <lookahead>.
//
//
void Flex::read_next(void)
{
  long	   p = 0;	      // Index within <lookahead>.
  char	   ch,		      // Temporary.
	   ch_next;	      // Temporary.
  boolean  hex = false;	      // Is this an integer in hex format?
  byte	   state;	      // State used when reading real numbers.
  boolean  got_char = false;  // Have we got a non-whitespace char?
  boolean  combine;	      // Combine 2 punctuation chars into one symbol?

  token_lines   = lines;
  token_columns = columns;

  space_before	= space_after;


  // Read a non-whitespace character into <ch>.
  //
  while (not got_char)	// While we don't have a non-whitespace character...
    {
    ch = next();	// Get the next character.

    switch (ch)
      {
      // If it is whitespace then ignore it...
      //
      case '\n':	// Linefeeds.
      case '\t':	// Tabs.
      case ' ' :	// Spaces.
      case '\r':	// Carriage Returns.
      case '\f':	// Formfeeds.
	space_before = true;
	break;

      // If it is a slash it could be the start of a comment.
      //
      case '/':
	if (not (syntax bitand flex_syntax_nocomment))
	  {
	  token_lines   = lines;
	  token_columns = columns;

	  ch_next = next();	  // Get the next character.

	  // Double Slash Comment (C++ Style).
	  //
	  // If it is also a slash then we have a double slash comment.
	  // We will skip everything to the end of the line or EOF.
	  //
	  if (ch_next == '/')
	    loop
	      {
	      ch_next = next();	  // Get the next character.
	      if (ch_next == '\n')  // If it is the end of the line
		break;		  // Then leave.

	      if (ch_next == 0)	  // If it is the EOF.
		{
		ch       = 0;
		got_char = true;  // It counts as a non-whitespace character.
		break;		  // So leave.
		}
	      }
	  //
	  // Single Slash and Star Comment (C Style).
	  //
	  // If it is search for a terminating asterix and slash.
	  //
	  else if (ch_next == '*')
	    {
	    loop
	      {
	      // Search for a terminating asterix or EOF.
	      //
	      do
		ch_next = next();
	      while (ch_next != '*' and ch_next != 0);

	      if (ch_next != 0)		// If not already EOF...
		ch_next = next();		// Then get the next character.

	      // If we have hit the EOF then register a parsing error.
	      //
	      if (ch_next == 0)
		parse_error("Unterminated /* ... */ Comment");
	      //
	      // If this is a slash we have reached the end of the comment.
	      //
	      else if (ch_next == '/')
		break;
	      //
	      // The asterix wasn't followed by a terminating slash; so this
	      // isn't the end of the comment.  The search continues...
	      // however if it was followed by another asterix we must hold
	      // it over since it may be the true end of the comment.
	      //
	      //
	      else if (ch_next == '*')
		reject_next();
	      }
	    }
	  else
	    {
	    // The second character wasn't a slash so this isn't a comment.
	    // We reject the second character, pushing it back into the
	    // input stream.  We set <got_char> to true because a slash
	    // by itself is a non-whitespace character.
	    //
	    reject_next();
	    got_char = true;
	    }
	  }
	else
	  got_char = true;
	break;


      case '\'':
	if (string_enclosure)
	  {
	  token_lines   = lines;
	  token_columns = columns;

          if (syntax bitand flex_syntax_lit_char)
	    {
	    if ((ch_next = next()) == '\\')
	      ch_next = next();

	    if (ch_next < ' ')
	      parse_error("Literal character must be printable");

	    if (next() != '\'')
	      parse_error("Closing single quote missing "
			  "from literal character");

	    lookahead_type = flex_char;
	    lookahead[0]   = ch_next;
	    lookahead[1]   = 0;

	    return;
	    }
	  }
	/*FALLTHROUGH*/


      default:
	//
	// FIX: Was incorrectly negated. [bj 09oct2006]
	//
	if (appl_comment_char != 0 and appl_comment_char == ch)
	  {
	  loop
	    if ((ch_next=next()) == '\n')
	      break;
	    else if (ch_next == 0)
	      {
	      ch       = 0;
	      got_char = true;  // It counts as a non-whitespace character.
	      break;
	      }

	   continue;
	   }


	//
	// Anything else counts as a non-whitespace character.
	//
	got_char = true;

	token_lines   = lines;
	token_columns = columns;
      }
    }


  // We have a character.  Store it at the start of the <lookahead> buffer.
  //
  lookahead[p++] = ch;


  // Check for a negative number.
  //
  // If we just read a '-' it may be prefixing a number.  If a digit
  // follows add it to the token and process the remainder just like
  // a positive number.  If not, hold over the character until next
  // time.
  //
  if (ch == '-')
    {
    ch_next = next();				// Get the next character.

    if ('0' <= ch_next and ch_next <= '9')	// If a digit...
      lookahead[p++] = ch = ch_next;		// Add it to the token.
    else
      // The character isn't a digit, so reject it (hold it over).
      //
      reject_next();
    }


  // Check for a hex number.
  //
  // Hex numbers consist of the sequence `0x' followed by the
  // hexadecimal value.
  //
  //
  else if (ch == '0')
    {
    ch_next = next();	    		// Get the next character.

    if (ch_next == 'x')
      {
      lookahead[p++] = ch = ch_next;	// Add it to the token.
      hex = true;    	        	// This is a hex value.
      }
    else
      // The character wasn't an `x', so reject it (hold it over).
      //
      reject_next();
    }



  // Read the rest of the token into <lookahead>.
  //

  // If the first character is a digit we have an integer or real number.
  //
  // We process this using the <state> variable.  We start in <state> 0.
  // If we encounter a `.' we moved into <state> 1.  If we encounter
  // an `E' in either <state> 0 or 1 we move into <state> 2.  In <state>
  // 2 the first character can be a `+' or `-'.  We then switch straight
  // to state <3> in which all the remaining characters must be digits.
  // If we never get past <state> 0 then we have an integer.  Anything
  // else means we have a real.
  //
  // number ::= digits [`.' [digits]] [`E' [`+'|`-'] digits]
  // state      0        1                  2        3
  //
  Select(('0' <= ch and ch <= '9')
	 or   (hex and ch == 'x'))
      state	     = 0;
      loop
	{
	ch = next();	// Get the next character.


	// Look for chracters in the correct state, or characters which
	// cause a transition between states.
	//
	if ((state == 0 or state == 1 or state == 3)
	     and '0' <= ch and ch <= '9')
	  ;
	else if (state == 0 and hex and 'a' <= ch and ch <= 'f')
	  ;
	else if (state == 0  and ch == '.' and not hex and
		not (syntax bitand flex_syntax_no_real))
	  state = 1;
	else if ((state == 0 or  state == 1) and not hex
		 and (ch == 'e' or ch == 'E'))
	  state = 2;
	else if (state == 2 and (ch == '+' or ch == '-'))
	  state = 3;
	else if (state == 2 and ('0' <= ch and ch <= '9'))
	  state = 3;
	else
	  {
	  // If the character doesn't fit the state then we've hit the
	  // end of the token.  Reject the non-conforming character
	  // and leave.
	  //
	  reject_next();
	  break;
	  }

	// Add the character to the token, making sure there is room.
	//
	if (p >= flex_token_size)
	  parse_error("String too long");
	else
	  lookahead[p++] = ch; // Add this character to the number.
	}


      // We have an integer iff we finished in <state> 0.
      // Anything else means we have a real number.
      //
      lookahead_type = (state > 0) ? flex_float :
		       (hex ? flex_hex : flex_int);


      // For a hexadecmal number we must have got at least three
      // characters in total.
      //
      if (hex and p < 3)
	parse_error("Hex value must follow `0x'");




    // If the first character is an alphanumeric or an underscore then
    // we have an identifier.
    //
    when('a' <= ch and ch <= 'z' or
	 'A' <= ch and ch <= 'Z' or ch == '_')
      loop
	{
	// Get the next character.
	//
	ch = next();

	// If it is alphanumeric, an underscore or a digit then append
	// it to the token.
	//
	if ('a' <= ch and ch <= 'z' or
	    'A' <= ch and ch <= 'Z' or
	    '0' <= ch and ch <= '9' or ch == '_')
	  lookahead[p++] = ch;
	else
	  {
	  // Otherwise we've hit the end of the token, so reject the
	  // unused character and leave the loop.
	  //
	  reject_next();
	  break;
	  }
	}
      lookahead_type = flex_id;	// This is an identifier.




    // If we have a double quote then we have a string.
    //
    when(ch == '"' and string_enclosure)
      p = 0;	      		 // Reset the buffer to discard the quote.
      loop
	{
	ch = next();		 // Get the next character.

	// If a backslash escape the next character.
	//
	if (ch == '\\' and string_conversion)
	  {
	  ch = next();

	  switch (ch)
	    {
	    case '\n': continue;	   // String continued on next line.
	    case 'n':  ch = '\n'; break;   // A new line.
	    case 'r':  ch = '\r'; break;   // A carriage return.
	    case 't':  ch = '\t'; break;   // A Tab.
	    case 'b':  ch = '\b'; break;   // A backspace.
	    case 0:    parse_error("Premature EOF in string"); 	// EOF.
	    default:;			   // Anything else is itself.
	    }
	  }
	else if (ch == '"')	 // If it is a double quote we've finished.
	  break;
	else if (ch == '\n' and not (syntax bitand flex_syntax_linestrings))
	  parse_error("Unescaped newlines not allowed in a string");
	else if (ch == 0)
	  parse_error("Premature EOF in string"); 		// EOF.


	// Add the character to the token, making sure there is room.
	//
	if (p >= flex_token_size)
	  parse_error("String too long");
	else
	  lookahead[p++] = ch; // Store this character in the string.
	}

      // We have a string.
      //
      lookahead_type = flex_string;




    // If we have a single quote then we have a string.
    //
    when(ch == '\'' and string_enclosure)
      p = 0;                   // Reset the buffer to discard the quote.
      loop
	{
	ch = next();           // Get the next character.

	if (ch == '\'')        // If it is a single quote we've finished.
	  break;
	else
	  lookahead[p++] = ch; // Store this character in the string.
	}

      // We have a string.
      //
      lookahead_type = flex_string;



    // A HTML escape; "&symbol[;]" or "&#NNN[;]"
    //
    when ((syntax bitand flex_syntax_amp_escape) and ch == '&')
      loop
	{
	ch = next();           // Get the next character.

	if (isspace(ch) or ch == 0)  // If whitespace we're finished.
	  {
	  reject_next();  // Don't consume the whitespace.
	  break;
	  }
	else
	  {
	  lookahead[p++] = ch; // Store this character in the string.

	  if (ch == ';')  // If the terminating semicolon we're finished.
	    break;
	  }
	}

      // We have a HTML escape token.
      //
      lookahead_type = flex_escape;



    // If zero we have the end of the file.
    //
    when(ch == 0)
      lookahead_type = flex_null;	// Set the token type to flex_null.


    // Anything else must be a punctuation symbol.
    //
    otherwise
      lookahead_type = flex_punc;	// A punctuation symbol.


      // Do special syntax processing if other than <flex_syntax_default>.
      //
      if (syntax)
	{
	// Some syntaxes have symbols made up by combining two punctuation
	// characters into one.  See if this is the case and if it is set
	// <combine> to true.
	//
	ch_next  = next();	// Look at the next character.
	combine  = false;	// Assume we won't be combining.


	// Equal Syntax: <= >= == != << >>
	//
	if (syntax bitand flex_syntax_equal)
	  {
	  switch (ch)
	    {
	    case '<':
	    case '>':
	    case '=':
	    case '!':
	      combine = ch_next == '=';
	      /*FALLTHROUGH*/

	    default:;
	    }

	  if (not combine)
	  switch (ch)
	    {
	    case '<':
	    case '>':
	      combine = ch == ch_next;
	      /*FALLTHROUGH*/

	    default:;
	    }
	  }


	// Goal Syntax: :- or ::
	//
	if (not combine and (syntax bitand flex_syntax_goal))
	  if (ch == ':')
	    combine   = ch_next == '-' or ch_next == ':';

	// Assign Syntax: :=
	//
	if (not combine and (syntax bitand flex_syntax_assign))
	  if (ch == ':')
	    combine   = ch_next == '=';


	// Assign Syntax: :=
	//
	if (not combine and (syntax bitand flex_syntax_assign_op))
	  if (ch == '+' or ch == '-' or ch == '*' or ch == '/' or ch == '%')
	    combine   = ch_next == '=';


	// Assign Syntax: Support ++, --.  [bj 21sep1998]
	//
	if (not combine and (syntax bitand flex_syntax_assign_op))
	  if (ch == '+' or ch == '-')
	    combine   = ch_next == ch;



	// Combine the second character iff <combine> is true,
	// otherwise hold the next character over for next time.
	//
	if (combine)
	  lookahead[p++] = ch_next;	// Then add it to the token.
	else
	  reject_next();	  	// Otherwise reject it.
	}
  endsel


  // Do a final check to make sure the token didn't overflow.
  //
  // TODO: We should really check this character by character,
  //       otherwise by the time we detect it could be too
  //       late.  [bj 17apr96]
  //
  if (p >= flex_token_size)
    parse_error("Token too long; overflow.");


  // Seal the token with a terminating null.
  //
  lookahead[p] = 0;


  // Look for trailing whitespace.
  //
  switch (ch=next())
    {
    // If it is whitespace then ignore it...
    //
    case '\n':	// Linefeeds.
    case '\t':	// Tabs.
    case ' ' :	// Spaces.
    case '\r':	// Carriage Returns.
    case '\f':	// Formfeeds.
      space_after = true;
      break;

    default:
      reject_next();
      space_after = false;
    }
}




// FUNCTION Flex Get.
//
// Get and return the next token.
//
// OUT:
//   S		Optional buffer into which to copy the token.
//		Set NULL if not required.
// RESULT:
//   The next token.
//
//
char *Flex::get(char *S, int Size)
{
  if (lookahead_type == flex_null)		// If at EOF...
    parse_error("Unexpected end of file");	// Register a parsing error.
  else
    {
    if (S == NULL)		    // If they didn't give us a buffer...
      S = lookahead_copy;	    // Then use our own <lookahead_copy>.

    if (Size < 0)
      strcpy(S, lookahead);	    // Store the token in the buffer.
    else
      strncpy(S, lookahead, Size);  // Store the token in the buffer.

    read_next();		    // Get the next token into <lookahead>.
    }

  return S;
}




// FUNCTION Flex Get Int.
//
// Return the next token which must be an integer (base 10 or hex).
//
// RESULT:
//   The next token as an integer.
//
//
long Flex::get_int(void)
{
  long result;


  switch (lookahead_type)
    {
    case flex_int:
    case flex_hex:
      result = (long)strtoul(lookahead, NULL,
			     lookahead_type == flex_int ? 10 : 16);
      read_next();
      break;


    default:
      parse_error("Integer (decimal or hexadecimal) expected");
      result = 0;
    }


  return result;
}




// FUNCTION Flex Get Double.
//
// Return the next token which must be a real number.
//
// RESULT:
//   The next token as a real number.
//
//
double Flex::get_double(void)
{
  double result;


  if (lookahead_type != flex_float and lookahead_type != flex_int)
    {
    parse_error("Real number expected");
    result = 0.0;
    }
  else
    {
    result = atof(lookahead);

    read_next();
    }


  return result;
}




// FUNCTION Flex Get Id.
//
// Return the next token which must be an identifier.
//
// RESULT:
//   The next token.
//
//
char *Flex::get_id(char *S, int Size)
{
  if (lookahead_type == flex_id)
    return get(S, Size);
  else
    {
    parse_error("Identifier expected");
    return NULL;
    }
}




// FUNCTION Flex Get String.
//
// Return the next token which must be a string.
//
// RESULT:
//   The next token.
//
//
char *Flex::get_string(char *S, int Size)
{
  if (lookahead_type == flex_string)
    return get(S, Size);
  else
    {
    parse_error("String expected");
    return NULL;
    }
}




// FUNCTION Flex Get Char.
//
// Return the next token which must be a character.
//
// RESULT:
//   The next token.
//
//
char Flex::get_char(void)
{
  char	result;

  if (lookahead_type == flex_char)
    {
    result = lookahead[0];

    read_next();

    return result;
    }
  else
    {
    parse_error("Literal character expected");
    return 0;
    }
}




// FUNCTION Flex Matches.
//
// Return true iff the next token matches <S> to a minimum of
// <Minimum> characters.  If it does match the next token
// is consumed.
//
// IN:
//   S		The string the next token must match.
//   Minimum	The minimum number of digits required to match.
// RESULT:
//   True iff it matches.
//
//
boolean Flex::matches(char *S, byte Minimum)
{
  boolean matches;	// Do we have a match?
  char	  *pS,		// Pointer to source template
	  *pN;		// Pointer to the next token.


  for (pS = S, pN = peek(); Minimum > 0; pS++, pN++, Minimum--)
    if (*pS != *pN)
      break;

  matches = Minimum <= 0;

  if (matches)
    get(NULL);


  return matches;
}




// FUNCTION Flex Matches.
//
// Return true iff the next token matches <S>.  If it does
// match the next token is consumed.
//
// IN:
//   S		The string the next token must match.
// RESULT:
//   True iff it matches.
//
//
boolean Flex::matches(char *S)
{
  boolean matches;			// Do we have a match?


  matches = equal(peek(), S);
  if (matches)
    read_next();


  return matches;
}




// PROCEDURE Flex Check.
//
// Check that the next token matches <S> to a minimum of
// <Minimum> characters.  If it does match the next token
// is consumed.  If it does not match a parsing error is
// registered.
//
// MOD003: Strings are not accepted.
//
// IN:
//   S		The string the next token must match.
//   Minimum	The minimum number of digits required to match.
//
//
void Flex::check(char *S, byte Minimum)
{
  if (lookahead_type == flex_string)
    parse_error("Token expected, but string found instead.", S);
  else if (not matches(S, Minimum))
    parse_error("Token expected but not found", S);
}




// PROCEDURE Flex Check.
//
// Check that the next token matches <S>.  If it does match the
// next token is consumed.  If it doesn't match a parsing error
// is registered.
//
// MOD003: Strings are not accepted.
//
// IN:
//   S		The string the next token must match.
//
//
void Flex::check(char *S)
{
 if (lookahead_type == flex_string)
    parse_error("Token expected, but string found instead.", S);
 else if (not matches(S))
    parse_error("Token expected but not found", S);
}





// PROCEDURE Flex Skip End of Line.
//
// Discard the current token and skip to the end of the current line.
//
//
void Flex::skip_eol(void)
{
  char		ch;


  do
    ch = next();
  while (ch != '\n' and ch != 0);


  read_next();
}





// PROCEDURE Flex Parse Error.
//
// Register a parsing error.
//
// What happens depend on the setting of the <abort> boolean member.
// If true then an error message is printed describing the error after
// which the program exits.  If false then the <error> flag is set to true
// and the parsing continues.  In the false case it is the responsibility
// of the application to check the <error> flag and initiate its own
// error handling routine.
//
// IN:
//   Message		A description of the error.
//   Token		An optional string printed with the error message.
//			NULL if unused.
// ABENDS:
//   Only if <abort> is true.
//
//
void Flex::parse_error(const char *Message, char *Token)
{
  short		i;

  if (abort)
    {
    cout  << flush;
    cerr  << "A Parsing Error has occurred:\n  ";

    if (Token)
      cerr << "\"" << Token << "\" ";

    cerr  << Message << "\n\n"
	  << "Where:  Around line " << lines+1 << " in column " << columns+1;

    if (source[0])
      cerr << " in " << source;
    cerr << nl;


    cerr  << "Text:  ^^";
    for (i=0; i<5 and not eof(); i++)
      if (not eof())
	cerr << " " << get();
    cerr << nl;

    exit(8);
    }
  else
    error++;
}




// PROCEDURE Flex Parse Warning.
//
// Register a parsing warning.
//
// Print a warning message indicating our current position within the
// source file.  This procedure returns after which the caller may
// continue parsing.
//
// IN:
//   Message		A description of the error.
//   Token		An optional string printed with the error message.
//			NULL if unused.
//
//
void Flex::parse_warning(const char *Message, char *Token)
{
  short		i;

  if (abort)
    {
    cout  << flush;
    cerr  << "Warning: " << Message << "\n"
	  << "Where:    Around line " << lines+1 << " in column " << columns+1;

    if (source[0])
      cerr << " in " << source;
    cerr << nl;

    cerr  << "Text:  ^^";
    if (Token)
      cerr << Token << " ";
    cerr << lookahead << "\"\n";
    }
  else
    warning++;
}




#ifdef TEST_FLEX
void main(void)
{
  Flex	F;

  F.open_string("-1.7e10 22 -33 12 :--::= <= 3.0\'Gossamer Thread\' How now brown cow",
  NULL, flex_syntax_ppl);

  cout << "Begin\n";

  cout << F.get_double() << nl;
  cout << F.get_int() << nl;
  cout << F.get_int() << nl;

  while (not F.eof())
    {
    cout << F.get() << nl;
    }
}
#endif
