//
// NAME:
//   rays.h
// TITLE:
//   Rays: Resizable arrays with optional bounds checking.
// FUNCTION:
//   Rays objects consists of a pointer to an array of elements and
//   a count of the number of elements in this array.
//
// OBJECTS:
//   Mantaray	A resizable array with optional bounds checking.
//   Stingray	Designed for pointers.
//		- Returns a constant pointer when a constant [] is used.
//		- Purges array elements during deconstruction.
// METHODS:
//   [], [] const	     Access elements just like a normal array
//   size(), size(N)	     Query or set the number of elements.
//   Stingray::purge()	     Deletes array elements contents.
//   Stingray::clean()	     Removes pointers to array elements.
//   add()	     	     Creates a new element, adding it to the list.
// TUNABLE DEFINES:
//   STINGRAY_NOBOUNDSCHECK  Turns off bounds checking.
//   NDEBUG		     Automatically selects STINGRAY_NOBOUNDSCHECK.
// PREREQUISITES:
//   Commmon.
//
// AUTHOR:
//   Brendan Jones. (Contact through www.kdef.com/geek/vic)
// RIGHTS:
//   Copyright Brendan Jones, 1994-1999.
// SECURITY:
//   Unclassified.
// LEGAL NOTICE:
//   See legal.txt before viewing, modifying or using this software.
// CONTACT:
//   Web:	http://www.kdef.com/geek/vic
//   Email:	See www.kdef.com/geek/vic
// DATE:
//   October 12, 1994.
// RIGHTS:
//  This file is part of The Quetzalcoatl Compiler.
//  
//  The Quetzalcoatl Compiler is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//  
//  The Quetzalcoatl Compiler is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//  
//  You should have received a copy of the GNU General Public License
//  along with The Quetzalcoatl Compiler; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
//
// MODIFICATIONS:
//   NAME  MOD  DATE     DESCRIPTION
//   bj    1    25aug95  Converted from Dynarray.h
//   bj    2    11apr97  Converted from Rsarray.h
//   bj    3    19may97  Added add().
//   bj    4    23may97
//   bj    5    18aug97  Added pragmas to support GNU C++.
//   bj    6    23dec97  Added <Stingray_autoctor> which has the
//			 old add(void) call. This constructs the
//			 item being added, saving several lines
//			 of code.  Added for backward compatibility
//			 with Minotaur, but you can use it whereever
//			 you're adding objects with a parameterless ctor.
//   bj    7    18jan99  Added chunky add(), increase(), assignment.
//   bj    8  09oct2006  Removed Harry's #ifndef GCC conditionals; 
///			 these were logically inverted (ie. giving an
//			 error when the index was fine); all compilers
//			 now use the same code.
//			  
//
//
#ifndef RAYS_H
#define RAYS_H
#ifdef	__GNUC__
#pragma interface
#endif
#include "common.h"
#include "implemen.h"

#ifdef	NDEBUG
#define	RAY_NOBOUNDSCHECK
#endif


// TEMPLATE OBJECT Mantaray.
//
// A dynamically resizable array with optional bounds checking.
//
//
template <class Type, class Counter_type> class Mantaray
{
  public:
    //
    // Number of elements in this array.
    //
    Counter_type	items;

    // Pointer to an array of <items> elements.
    //
    Type	       *item;


  public:
    //
    // Construct an empty array.
    //
    Mantaray(void)
      {
      item  = NULL;
      items = 0;
      }


    // Destruct, deleting an array elements.
    //
   ~Mantaray(void)
      {
      if (item != NULL)
	delete[] item;
      };


    // CONSTANT OPERATOR [].
    //
    // This returns an element that may not be modified.
    //
    Type &operator[](const Counter_type i) const
      {
      #ifndef RAY_NOBOUNDSCHECK
       if (i < 0 or i >= items)
	abend(WHERE0, "Overflow");
      #endif

      return item[i];
      }


    // OPERATOR [].
    //
    // This returns an element that may be modified.
    //
    Type &operator[](const Counter_type i)
      {
      #ifndef RAY_NOBOUNDSCHECK
      if (i < 0 or i >= items)
	abend(WHERE0, "Overflow");
      #endif

      return item[i];
      }


    // Resize the array so it holds <Items>.
    //
    void size(Counter_type Items);


    // How many elements are there currently in this array?
    //
    Counter_type size(void) const
      {return items;}


    // Resize the array so it holds <Items>.
    //
    void increase(Counter_type Items)
      {size(size()+Items);}


    // Create a new element, adding it to the end.
    //
    Type *add(void)
      {
      Counter_type current_size = items;
      size(items+1);
      return &item[current_size];
      }


    // Create a new element at the <Current> position.
    // If the array is full, automatically expand it by <Chunk_size>.
    //
    Type& add(Counter_type Current, Counter_type Chunk_size)
      {
      if (Current+1 >= size())
        ;
      else
	increase(Chunk_size);

      return item[Current];
      }


    // Assignment.
    //
    Mantaray<Type, Counter_type>& operator =
	(const Mantaray<Type, Counter_type>& Source);
};




// FUNCTION Mantaray Size.
//
// Resize the array so it holds <Items>.
//
//
template <class Type, class Counter_type>
Mantaray<Type, Counter_type>& Mantaray<Type, Counter_type>::operator =
	(const Mantaray<Type, Counter_type>& Source)
{
  items	= Source.items;
  mcheck(item  = new Type [items]);

  for (Counter_type i=0; i<Source.items; i++)
    item[i] = Source.item[i];

  return *this;
}




// FUNCTION Mantaray Size.
//
// Resize the array so it holds <Items>.
//
//
template <class Type, class Counter_type>
void Mantaray<Type, Counter_type>::size(Counter_type Items)
{
  if (Items != items)
    {
    Type       		*temp;
    Counter_type      	 i;
    Counter_type      	 copy;

    if (Items == 0)
      temp = NULL;
    else
      {
      mcheck(temp = new Type [Items]);

      copy = min(Items, items);
      for (i=0; i<copy; i++)
	temp[i] = item[i];
      }

    if (item != NULL)
      delete[] item;

    item  = temp;
    items = Items;
    }
};




template <class Type, class Counter_type> class Stingray
	: public Mantaray<Type *, Counter_type>
{
  public:
    // Clear the pointers to elements in the array,
    // without actually deleting them.
    //
    void clear(void);


    // Purge the elements in array, deleting each of them.
    //
    void purge(void);


    // CONSTANT OPERATOR [].
    //
    // This returns an element that may not be modified.
    //
    const Type &operator[](const Counter_type i) const
      {
      #ifndef RAY_NOBOUNDSCHECK
       if (i < 0 or i >= items)
	abend(WHERE0, "Overflow");
      #endif

      return *item[i];
      }


    // OPERATOR [].
    //
    // This returns an element that may be modified.
    //
    Type &operator[](const Counter_type i)
      {
      #ifndef RAY_NOBOUNDSCHECK
      if (i < 0 or i >= items)
	abend(WHERE0, "Overflow");
      #endif

      return *item[i];
      }


    // How many elements are there currently in this array?
    //
    Counter_type size(void) const
      {return items;}


    // Resize the array so it holds <Items>.
    //
    void size(Counter_type Items);


    // Add a new element slot to the end.
    // Return a pointer to this slot.
    // The caller can then fill it in.
    //
    Counter_type add(Type *Entry)
      {
      Counter_type current_size = items;
      size(items+1);

      item[current_size] = Entry;

      return current_size;
      }


    // Resize the array so it holds <Items>.
    //
    void increase(Counter_type Items)
      {size(size()+Items);}


     //
     //
     Type *set(Counter_type Item, Type *Value)
       {
       return item[Item] = Value;
       }


    boolean occupied(Counter_type Item) const
      {
      return item[Item] != NULL;
      }


    boolean empty(Counter_type Item) const
      {
      return item[Item] == NULL;
      }


  public:
   ~Stingray(void)
      {
      purge();
      }
};



// FUNCTION Stingray Size.
//
// Resize the array so it holds <Items>.
//
//
template <class Type, class Counter_type>
void Stingray<Type, Counter_type>::size(Counter_type Items)
{
  if (Items != items)
    {
    Type       	       **temp;
    Counter_type      	 i;
    Counter_type      	 copy;

    if (Items == 0)
      temp = NULL;
    else
      {
      mcheck(temp = new Type * [Items]);

      copy = min(Items, items);
      for (i=0; i<copy; i++)
	temp[i] = item[i];

      for (i=copy; i<Items; i++)
        temp[i] = NULL;
      }

    if (item != NULL)
      delete[] item;

    item  = temp;
    items = Items;
    }
};




// PROCEDURE Stingray Clear.
//
// Clear the pointers to elements in the array,
// without actually deleting them.
//
//
template <class Type, class Counter_type>
void Stingray<Type, Counter_type>::clear(void)
{
  Counter_type	i;


  for (i=0; i<items; i++)
    item[i] = NULL;
};






// PROCEDURE Stingray Purge.
//
// Delete elements in array.
//
//
template <class Type, class Counter_type>
void Stingray<Type, Counter_type>::purge(void)
{
  Counter_type	i;


  for (i=0; i<items; i++)
    if (item[i] != NULL)
      {
      delete item[i];

      item[i] = NULL;
      }
};


//
//
//
template <class Type, class Counter_type> class Stingray_autoctor
	: public Stingray<Type, Counter_type>
{
  public:
    // Add a new element slot to the end.
    // Return a pointer to this slot.
    // The caller can then fill it in.
    //
    Type *add(void)
      {
      Type *Entry = new Type;
      mcheck(Entry);

      Counter_type current_size = items;
      size(items+1);

      item[current_size] = Entry;

      return Entry;
      }
};






template <class Type> class Stingray_short : public Stingray<Type, short> {};
template <class Type> class Stingray_long  : public Stingray<Type, long>  {};
template <class Type> class Mantaray_short : public Mantaray<Type, short> {};
template <class Type> class Mantaray_long  : public Mantaray<Type, long>  {};
template <class Type> class Stingray_autoctor_short : public Stingray_autoctor<Type, short> {};
template <class Type> class Stingray_autoctor_long  : public Stingray_autoctor<Type, long>  {};
#endif
