//
//
// NAME:
//   uplrtime.asm
// TITLE:
//   UPL Runtime Library.  (Version 2.1.0 BETA)
// FUNCTION:
//   The runtime library provides basic I/O, stack management,
//   logical and arithmetic (8-bit and 16-bit, signed and unsigned)
//   services to programs compiled with the Queztalcoatl compiler
//   for the 6502 family of processors.
//
// TUNABLE DEFINES:
//   target		Selects runtime library target.
//			eg. -Dtarget=1 for Unexpanded VIC-20.
//
// PORTING:
//   Porting this runtime library to other 6502 platforms should be
//   fairly straight forward.  The compiler assumes the runtime
//   library has CL, CH and BP pseudo-registers in a certain
//   location, and that <datastack> is allocated an entire page
//   and that the current stack frame is indexed with BP.
//   Obviously the called routines must do exactly what they
//   say they'll do.  Apart from these conditions, the compiler
//   doesn't really care about the runtime library's internals.
//
//   To port the library to another platform.
//     0.  Assign yourself a target number.  Check the target enumerations
//         below.  If your platform is already there, use that number.
//         If not, assign yourself a new target number.  When you mofify
//         this runtime library you can include code just for your target
//         by using the if (target == target_gizmo) conditional,
//         where gizmo is the target you are adding.
//     1.  Change org to reflect the start of the application memory area.
//     2.  Move whatever registers and variables you can into zero-page.
//         (This generic implementation was written assuming only a few
//          zero-page locations were available.)
//     3.  Change the kernel routines for console I/O.  If your target
//         platform doesn't have corresponding routines that do this
//         (eg. correct handling for newlines), you'll need to write
//         filters for them.  On some targets this may not be possible;
//	   eg. small handhelds or game consoles without keyboards.
//     4.  That's it!  Recompile the runtime library, saving the
//         output under name name uplrXXXX.obj, where XXXX is the
//         name of the target platform. eg. uplrlynx.obj for an Atari Lyynx.
//         eg. quetzal -Dtarget=5 -cm uplrlynx.obj uplrtime.asm
//         If you look at the target enumerations below, you'll
//         see that 5 is the target number I've assigned to the lynx.
//         If you're porting to another platform you'll need another
//         target number.
//     5.  When compiling programs to use your new target include the
//         option -r lynx on the linker command line.  It'll link in
//         your runtime library in rather than the default one.
//     6.  Send me a copy and I'll place it online at
//	   http://www.kdef.com/geek/vic for other Quetzalcoatlites.
//         Please do this within a reasonable timeframe;  The runtime
//         library is updated from time to time, and if your version
//         is six months out of date by the time I get it it could
//         make merging it with the current version difficult.
//
// NOTE: The "and" opcode.
//   The Quetzalcoatl assembler is a free-format assembler.
//   That is, an instruction can spread over several lines.
//   It also evaluates complex conditional expressions, such
//   as those containing *, /, +, -, "not", "or" and "and".
//   The latter produces some problems, because it just happens
//   to be a 6502 opcode.  This means "cmp #fred and wilma"
//   (even when spread over two lines) is interpreted as *one*
//   instruction.  To break it up you need to place a semicolon ";"
//   after the cmp line.  Thus "cmp #fred; and wilma".  This
//   problem only happend with the "and" opcode.  While it is
//   easy to fix, it's still annoying an easy to overlook.
//   In a future version I'll find a better way to fix this.
//   (Perhaps restricting instructions to a single line)
//   [bj 21sep1998]
//
// PREREQUISITES:
//   None.
//
// AUTHOR:
//   Brendan Jones, Kestrel Defence.
// RIGHTS:
//   (c) Copyright Kestrel Defence, 1998.  All Rights Reserved.
//
//   You're encouraged to modify this library to optimise it,
//   port it to other platforms or to produce customised versions
//   for a particular application (for example, omitting functions
//   that are not used by that application).  Modifications should
//   be clearly labelled and the file should be labelled such that
//   is not confused with the original version.  Apart from an
//   entry under the MODIFICATIONS change log below, the text of
//   this header and the accompanying legal notice must be
//   distributed with it verbatim.
//
//   Use of the Quetzalcoatl compiler is subject to a separate
//   licence agreement, contained in the file legal.txt.
//   You must read and agree to the terms of the licence before
//   using this software.
//
// WEB:
//   http://www.kdef.com/geek/vic
//
// SECURITY:
//   Unclassified.  For public distribution.
// CREATION DATE:
//   July 20, 1998.
//
// MODIFICATIONS:
//   NAME  MOD  DATE       DESCRIPTION
//   bj    1    21sep1998  Alpha Release 2.0a1.
//   bj    2    22sep1998  Fixed bugs; shaved off a whopping 33 bytes! :/
//			   Added conditionals so one runtime library source
//   bj    3    24sep1998  Renamed mul_ww umul_ww (which it really is).
//			   Added but not optimised a new mul_ww.
//			   Added vicg target.
//   bj    4    25sep1998  Added vich target.
//   bj    5    30sep1998  Added c64  target.
//   bj    6     7oct1998  Fixed bug: neg_w.
//   hfd        18apr2005  Added Rockwell AIM-65 support
//   hfd        20apr2005  many many fixes/speedups/additions
//   hfd        22apr2005  added divide by 0 test to avoid infinite loop
//   hfd        26apr2005  added Atari 7800 support
//			   added ROM support via a dataseg define and movemem routine
//   hfd        27apr2005  added fast divide by 256 - could move to upl_copt at some point
//			   optimized peek_w and poke_w for speed
//   hfd	04may2005  added routines to extract hi/lo bytes of integers
//			   added routine to return the remainder of the previous division
//			   added routine for extracting the address of a variable (safely)
//		05may2005  split this into two files
//		12may2005  added memmove() and memset() routines
//			   fixed signed division and remainder

		// GROUP Target.
		//

		// ENUMERATION target_*.
		//
		// Each constant specifies a different target platform.
		// These may be selectd with eg. -Dtarget=2 for vicx.
		//
		target_generic	= 0
		target_vicu	= 1	// VIC-20 with 3.5Kb RAM
		target_apple2	= 2	// Apple ][ with 48Kb RAM.
		target_vicx	= 3	// VIC-20 with >= 11.5Kb RAM
		target_c64	= 4	// Commodore 64
		target_lynx	= 5	// Not implemented.
		target_nes 	= 6     // Not implemented.

		// Graphical VIC-20 with >= 11.5Kb RAM.
		//
		// This is a variation of vicx, but with a memory
		// configuration better suited to bitmapped graphics.
		//
		// The character set begins at           0x1000.
		// The video frame buffer begins at      0x1800.
		// The runtime library proper begins at  0x1A00.
		// When we start running BASIC begins at 0x1200.
		//
		// If you place the BASIC bootstrap at 0x1200
		// you'll lose 16 bytes (two characters); 64 and 65.
		//
		target_vicg	= 7	// Graphical Expanded VIC.


		// Full-screen graphical VIC-20 with >= 11.5Kb RAM.
		//
		// This is a variation of vicx, but with a memory
		// configuration better suited to full-screen bitmapped
		// graphics.
		//
		// The video frame buffer stays at       0x1000.
		// A 512 byte unused block of memory     0x1200
		// The character set begins at           0x1400.
		// The runtime library proper begins at  0x2400.
		// When we start running BASIC begins at 0x1200.
		//
		target_vich	= 8	// Graphical Expanded VIC.
		target_aim	= 9	// AIM-65
		target_7800	= 10	// Atari-7800


		// If no target is defined default to the generic one.
		//
		if (defined(target) == 0)
		  target = target_generic;
		endif

		// Identify the target so there's no confusion.
		//
		if (target == target_generic)
		  message 1 "Target is Generic Commodore."
		elif (target == target_vicu)
		  message 1 "Target is Unexpanded VIC-20 with 3.5Kb RAM."
		elif (target == target_vicx)
		  message 1 "Target is Expanded VIC-20 with >= 11.5Kb RAM."
		elif (target == target_c64)
		  message 1 "Target is Commodore 64."
		elif (target == target_apple2)
		  error "Target is Apple ][ with >= 48Kb RAM. (NOT IMPLEMENTED)"
		elif (target == target_lynx)
		  error "Target is Atari Lynx. (NOT IMPLEMENTED)"
		elif (target == target_nes)
		  error "Target is Nintendo NES. (NOT IMPLEMENTED)"
		elif (target == target_vicg)
		  message 1 "Target is Bitmap-optimised Expanded VIC-20 with >= 11.5Kb RAM."
		elif (target == target_vich)
		  message 1 "Target is a Full-screen Bitmap-optimised Expanded VIC-20 with >= 11.5Kb RAM."
		elif (target == target_aim)
		  message 1 "Target is a Rockwell AIM-65."
		elif (target == target_7800)
		  message 1 "Target is an Atari 7800 Prosystem."
		else
                  error "Unspecified target: Define me!"
		endif


		// State Target.
		//
		// Record the target enumeration in the object file.
		// Although not currently used, in future versions
		// it could be used to automatically choose the best
		// runtime library for an application, and to prevent
		// the wrong runtime library accidentally being linked.
		//
		declare_target(target)
		declare_flags(0)		// Not yet used.

		// HFD really put the target into the runtime object file
		// used to choose the type of header that the compiler generates right now.
		// it could be used to have the compiler and optimizer generate 65c02
		// code as well.

		org target
machine:

		// HFD
		// set this so no data relocation at runtime

		if (target == target_vicx 	or
		    target == target_vicg 	or
		    target == target_vich 	or
		    target == target_vicu 	or
		    target == target_generic 	or
		    target == target_c64	or
		    target == target_aim)

		  org	0xffff
dataseg:
		endif

		// Starting Address.
		//
		// NOTE:
		//   Under Quetzalcoatl version 2.0a1 the assembler
		//   produces absolute rather than relocatable code.
		//   The linker places this absolute code at the
		//   very start of the executable.  Thus you should
		//   set the origin to the start of application memory.
		//   Later versions of the assembler will produce
		//   relocatable code.  Until then... [bj 21sep1998]
		//
		// IMPORTANT:
		//   The runtime library must start on a page boundary.
		//   That is, the least significant byte of the
		//   org starting address must be 0x00.
		//   !!! No, it doesn't !!!  HFD
		//
		if (target == target_aim)
		  cmos_cpu = 1
		else
		  cmos_cpu = 0
		endif

		if (target == target_aim)
		  org 0x0200
datastack:
		endif

		if (target == target_7800)
		  org	0x1800
datastack:
		  org	0x1900
dataseg:
		endif

// set start address

		if (target == target_vicu or
		    target == target_vicg or
		    target == target_generic)
		  org   0x1000

		// ---------------------------------------------------
		elif (target == target_aim)
		// set 6 bytes above where it loads to allow space for
		// the compiler to insert these 2 instructions:
		// JSR start_addr
		// JMP monit
		  org	0x1006

		elif (target == target_7800)
		// set 9 bytes high for compiler to add 
		// JMP start_addr
		  org	0xd003
params:
		// and 6 bytes of data segment relocation information
		  org	0xd009
		// Interrupt vectors
		// needs these where we can find them when making a ROM
		    dw IRQ
		    dw NMI
		// ---------------------------------------------------

		elif (target == target_vicx or
		      target == target_vich)
		  org   0x1200
		elif (target == target_c64)
		  org	0x0800
		else
		  error "Undefined target: Define me!"
		endif




		// GROUP Implementation Specific Kernel Routines.
		//

		// Print the character in A on standard output.
		if (target == target_vicx 	or
		    target == target_vicg 	or
		    target == target_vich 	or
		    target == target_vicu 	or
		    target == target_generic 	or
		    target == target_c64)
		// This routine is only called by the runtime
		// library from the routine print_ch_a, which
		// performs whatever corrections are required
		// (eg. newline translation that makes sure
		// '\n' 0x0a always moves the cursor to the
		// start of the next line).
		//
		call_print_ch_a 	= 0xffd2

		// Get Raw Character.
		//
		// Get a character from standard input and return it in A.
		// If no character is ready set A to zero and return
		// immediately.
		//
		call_get_ch_raw_a	= 0xffe4

		// Get Cooked Character.
		//
		// Get a character from standard input; the next character
		// from a line which the user has had a chance to edit
		// before pressing Enter.
		//
		call_get_ch_cooked_a	= 0xffcf

		elif (target == target_aim)
		  call_print_ch_a 	= 0xeeaa
		  call_get_ch_raw_a	= 0xebdb
		  call_get_ch_cooked_a	= 0xebdb

		elif (target == target_7800)
		  call_print_ch_a 	= a_rts	// no char I/O for 7800
		  call_get_ch_raw_a	= a_rts
		  call_get_ch_cooked_a	= a_rts

		else
		  error "Undefined target: Define me!"
		endif




		// GROUP Charmode Constants.
		//
		// These are enumerated constants we use to store
		// the current I/O control mode of reading standard input.
		// (eg. raw, cooked).
		//
		// The compiler uses the same values, so don't change them.
		//
		charmode_cooked		= 0x00	// DON'T CHANGE!
		charmode_raw_wait	= 0x01	// DON'T CHANGE!
		charmode_raw_nowait	= 0x80  // DON'T CHANGE!



		// GROUP Reserved.
		//

		if (target == target_vicg)
		//
		// Bitmap-optimised Expanded VIC-20.
		//
		// Allocate 2.5Kb.
		//
		// The first 2Kb of this (starting at 0x1000) will become
		// the new character set.  The existing video frame buffer
		// at 0x1000 will need to be moved to 0x1800 to make way.
		// The next 512 bytes begins at is this new video frame
		// buffer.
		//
		// The init routine will change the VICs registers to
		// conform with this new configuration.
		//
		//
		  db 0(2048)	// Room for an 8*8 Character set.  (2Kb)
		  db 32(512)	// New video frame buffer.
				// Initialise with spaces.

		elif (target == target_vich)
		//
		// Full-screen Bitmap-optimised Expanded VIC-20.
		//
		// Allocate 4.5Kb.
		//
		// The first 512 bytes is wasted space (at 0x1200),
		// since the character can't start between a 1Kb boundary.
		// We salvage what we can by using 256 bytes of that as
		// the datastack.  The remainder is unused.
		//
		// The 4Kb character set (at 0x1400) follows.
		//
		// The init routine will change the VICs registers to
		// conform with this new configuration.
		//
		//
datastack:	  db 0(256)	// Salvage wasted space.
		  db 0(256)	// Unused.
		  db 0(4096)	// Room for an 8*16 Character set.  (4Kb)
		endif

		// GROUP Data.
		//

		// SEGMENT Data.
		//
		// This section contains variables.  The only variables
		// that need to be here are the arrays such as
		// <datastack> and <buffer>.  To improve speed and
		// reduce the size of the runtime library you can
		// move other variables to free locations in the
		// zero page (if you have any).
		//


		// Datastack.
		//
		// Quetzacoatl uses a <datastack> to perform some
		// expression evaluation, automatic variables,
		// parameters and for temporary storage.
		//
		// In practice the compiler tries to avoid using
		// the stack whenever possible.
		//
		// IMPORTANT:
		// <datastack> *must* begin on a page boundary.
		//
		//
		if (target == target_vicx or
		    target == target_vicg or
		    target == target_generic or
		    target == target_vicu or
		    target == target_c64)
datastack:	  db 0(256)
		endif


		if (target == target_vicx or
		    target == target_vicg or
		    target == target_vich or
		    target == target_vicu)
		//
		// IMPLEMENTATION: VIC-20.
		//
		// Source: "VIC Revealed".  Nick Hampshire.  1981.  No ISBN.
		// (Caution: This book has contains errors/inconstencies.)
		//
		// The VIC's ROM has very few spare locations in page zero.
		// We use locations used by the ROM's RS232C routines;
		// (Obviously this means this configuration of the runtime
		// library cannot be run if you're doing an RS232C transfer.)
		// An alternative would be the BASIC numerical working area
		// between 87-96;  However this could break any programs
		// that make calls to BASIC routines, such as the Meteor
		// sample programs calls to the random number generator.
		//
		// The VIC also has some spare memory between 673-776.
		//
		// [bj 22sep1998]
		//
		//
		datastacks	= 167
		el		= 168
		eh		= 169
		fl		= 170
		fh		= 171
		charmode	= 180
		is_negative 	= 181
		bitcount	= 182

		elif (target == target_c64)
		//
		// IMPLEMENTATION: Commodore 64.
		//
		// Source funet.fi/pub/cbm/MemoryMaps.
		//
		// The C64 has fewer zero-page locations even than the VIC-20.
		// We use free locations higher in the memory map.  Note that
		// location 255 which we assume is the BP (Base pointer),
		// is actually used by the C64 for temporary BASIC data area.
		// This'd make calling a program using this runtime library
		// from BASIC (or BASIC routines) a tenuous proposition.
		//
		datastacks	= 1020
		el		= 1021
		eh		= 1022
		fl		= 679
		fh		= 680
		charmode	= 681
		is_negative 	= 682
		bitcount	= 683

		// ---------------------------------------------------
		elif (target == target_aim)
		// put at top of page zero beneath fixed registers
		el		= 247
		eh		= 248
		fl		= 249
		fh		= 250
		// put in a safe spot below AIM editor variables
		datastacks	= 0xd0
		charmode	= 0xd1
		is_negative 	= 0xd2
		bitcount	= 0xd3

		elif (target == target_7800)
		// put at bottom of page 0 RAM (above registers)
		el		= 64
		eh		= 65
		fl		= 66
		fh		= 67
		datastacks	= 68
		charmode	= 69
		is_negative 	= 70
		bitcount	= 71
		// ---------------------------------------------------

		else
		//
		// IMPLEMENTATION: Generic.
		//

		// Datastacks.
		//
		// An index to <datastack>, pointing to the top of the stack.
		//
datastacks:	db 0

el:		db 0		// Low  byte - Pseudo register E.
eh:		db 0    	// Low  byte - Pseudo register E.

fl:		db 0		// Low  byte - Pseudo register F.
fh:		db 0    	// Low  byte - Pseudo register F.

charmode:	db 0            // Current ioctl mode.

is_negative:	db 0            // Temporary.
bitcount:	db 0		// Temporary.

		endif


		// SUBGROUP Fixed Pseudo Registers.
		//
		// The compiler assumes these psuedo registers are
		// at the stated locations.  Do not move them!
		//
		// <C> must be immediately followed by <D>.
		//
		//
		cl = 251    // Low  byte - Pseudo register C. DON'T CHANGE!
		ch = 252    // High byte - Pseudo register C. DON'T CHANGE!
		dl = 253    // Low  byte - Pseudo register D. DON'T CHANGE!
		dh = 254    // High byte - Pseudo register D. DON'T CHANGE!
		bp = 255    // Base Pointer.                  DON'T CHANGE!

		include "runtime.asm"

		end
