/*	Sample code to remove register reloads
	HFD - June 30th 2005

	The goal is not to optimize all of them, but to avoid over-optimization
	which would break the program.  For example, as coded it will miss
	1)
	LDA value,X
	CMP #5
	BNE +12
	LDA value,X
	CMP #7
	...

	2)
	LDA value
	STA address1
	LDA value
	STA address2
	...

	These could be caught though with a bit more work:
	1 - have to verify that X hasn't changed between LDA instructions
	2 - have to track both last load and last store
 
	It will catch the following though
	LDA value1
	LDX value2
	STA address1
	STX address2
	LDA address1
	LDX address2
	...

*/

/*	Algorithm
First scan the object file for all branch/jump/subroutine destinations
Next go through the object file looking at instructions and skip the ones
  which reload the registers based on defined rules
Then rescan the resulting file and fixup destinations
*/

/*	Rules
if address=destination of branch or jmp or jsr
  unset all trackers

if instruction=jsr jmp rts
  unset all trackers

if addressing mode is indexed or indirect
  always keep to be safe

if instruction=lda
  if last_set_A = same value, remove it
  else last_set_A = value

if instruction=txa tya pla
unset last_set_A

if instruction=adc sbc rol ror asl lsr eor and ora
unset last_set_A

if instruction=tax tay inx iny dex dey tsx
unset last_set_X or Y

if instruction=sta
  last_set_A = value

if instruction=ldx
  if last_set_X = same value, remove it
  else last_set_X = value

if instruction=stx
  last_set_X = value

if instruction=ldy
  if last_set_Y = same value, remove it
  else last_set_Y = value

if instruction=sty
  last_set_Y = value

if instruction=bne beq bpl bmi and previous was lda ldx ldy
then keep previous

if instruction = bit
  ignore
*/



#include <stdio.h>
#define SCITE 1
#ifdef SCITE		// defines

#ifndef uint
#define uint unsigned int
#endif
#ifndef bool
#define bool char
#endif
#ifndef byte
#define byte unsigned char
#endif
#ifndef true
#define true 0xff
#endif
#ifndef false
#define false 0
#endif

#define ASM_BRK		0
#define ASM_ORA_IX	1
#define ASM_ORA_Z	5
#define ASM_ASL_Z	6
#define ASM_PHP		8
#define ASM_ORA_IMM	9
#define ASM_ASL_A	10
#define ASM_ORA		13
#define ASM_ASL		14
#define ASM_BPL		16
#define ASM_ORA_IY	17
#define ASM_ORA_ZX	21
#define ASM_ASL_ZX	22
#define ASM_CLC		24
#define ASM_ORA_Y	25
#define ASM_ORA_X	29
#define ASM_ASL_X	30
#define ASM_JSR		32
#define ASM_AND_IX	33
#define ASM_BIT_Z	36
#define ASM_AND_Z	37
#define ASM_ROL_Z	38
#define	ASM_PLP		40
#define	ASM_AND_IMM	41
#define ASM_BIT		44
#define ASM_AND		45
#define ASM_ROL		46
#define ASM_BMI		48
#define ASM_AND_IY	49
#define ASM_AND_ZX	53
#define ASM_ROL_ZX	54
#define ASM_SEC		56
#define ASM_AND_Y	57
#define ASM_AND_X	61
#define ASM_ROL_X	62
#define ASM_RTI		64
#define ASM_EOR_IX	65
#define ASM_EOR_Z	69
#define ASM_LSR_Z	70
#define ASM_PHA		72
#define ASM_EOR_IMM	73
#define ASM_LSR_A	74
#define ASM_JMP		76
#define ASM_EOR		77
#define ASM_LSR		78
#define ASM_BVC		80
#define ASM_EOR_IY	81
#define ASM_EOR_ZX	85
#define ASM_LSR_ZX	86
#define ASM_CLI		88
#define ASM_EOR_Y	89
#define ASM_EOR_X	93
#define ASM_LSR_X	94
#define ASM_RTS		96
#define ASM_ADC_IX	97
#define ASM_ADC_Z	101
#define ASM_RCR_Z	102
#define ASM_PLA		104
#define ASM_ADC_IMM	105
#define ASM_RCR_A	106
#define ASM_JMP_I	108
#define ASM_ADC		109
#define ASM_ROR		110
#define ASM_BVS		112
#define ASM_ADC_IY	113
#define ASM_ADC_ZX	117
#define ASM_ROR_ZX	118
#define ASM_SEI		120
#define ASM_ADC_Y	121
#define ASM_ADC_X	125
#define ASM_ROR_X	126
#define ASM_STA_IX	129
#define ASM_STY_Z	132
#define ASM_STA_Z	133
#define ASM_STX_Z	134
#define ASM_DEY		136
#define ASM_TXA		138
#define ASM_STY		140
#define ASM_STA		141
#define ASM_STX		142
#define ASM_BCC		144
#define ASM_STA_IY	145
#define ASM_STY_ZX	148
#define ASM_STA_ZX	149
#define ASM_STX_ZY	150
#define ASM_TYA		152
#define ASM_STA_Y	153
#define ASM_TXS		154
#define ASM_STA_X	157
#define ASM_LDY_IMM	160
#define ASM_LDA_IX	161
#define ASM_LDX_IMM	162
#define ASM_LDY_Z	164
#define ASM_LDA_Z	165
#define ASM_LDX_Z	166
#define ASM_TAY		168
#define ASM_LDA_IMM	169
#define ASM_TAX		170
#define ASM_LDY		172
#define ASM_LDA		173
#define ASM_LDX		174
#define ASM_BCS		176
#define ASM_LDA_IY	177
#define ASM_LDY_ZX	180
#define ASM_LDA_ZX	181
#define ASM_LDX_ZY	182
#define ASM_CLV		184
#define ASM_LDA_Y	185
#define ASM_TSX		186
#define ASM_LDY_X	188
#define ASM_LDA_X	189
#define ASM_LDX_Y	190
#define ASM_CPY_IMM	192
#define ASM_CMP_IX	193
#define ASM_CPY_Z	196
#define ASM_CMP_Z	197
#define ASM_DEC_Z	198
#define ASM_INY		200
#define ASM_CMP_IMM	201
#define ASM_DEX		202
#define ASM_CPY		204
#define ASM_CMP		205
#define ASM_DEC		206
#define ASM_BNE		208
#define ASM_CMP_IY	209
#define ASM_CMP_ZX	213
#define ASM_DEC_ZX	214
#define ASM_CLD		216
#define ASM_CMP_Y	217
#define ASM_CMP_X	221
#define ASM_DEC_X	222
#define ASM_CPX_IMM	224
#define ASM_SBC_IX	225
#define ASM_CPX_Z	228
#define ASM_SBC_Z	229
#define ASM_INC_Z	230
#define ASM_INX		232
#define ASM_SBC_IMM	233
#define ASM_NOP		234
#define ASM_CPX		236
#define ASM_SBC		237
#define ASM_INC		238
#define ASM_BEQ		240
#define ASM_SBC_IY	241
#define ASM_SBC_ZX	245
#define ASM_INC_ZX	246
#define ASM_SED		248
#define ASM_SBC_Y	249
#define ASM_SBC_X	253
#define ASM_INC_X	254
#endif

int main(void) {

#ifdef SCITE		// variables
	uint	i;

	uint	C_address;
	uint	P_address;
	byte	C_opcode = 0;
	byte	P_opcode;
	uint	C_operand;

	uint	reduction = 0;

	uint	addr_start;		// code segment start address
	uint	addr_end;		// code segment end address

	uint	X_val	= 0xffff;	// this address is almost always in ROM for 6502
	uint	Y_val	= 0xffff;
	uint	A_val	= 0xffff;
	bool	Keep[65536];		// true if keeping this instruction
	byte	Size[256];		// instruction size (1-3)		- needs filling
	bool	Mode[256];		// true if indexed or indirect mode	- needs filling
	char	Register[256];		// hold destination register A,X,Y	- needs filling
	byte	Program[65536];		// program loaded into run location?
	bool	Destination[65536];	// true if address is end of branch, jmp, or jsr 
#endif

	for (i=0;i<65536;i++) {
		Keep[i] = true;
		Destination[i] = false;
	}

/* read input file goes here
*/

/* scan for destination addresses goes here
*/

	C_address = addr_start;
	while (C_address < addr_end) {
		if (C_address != addr_start) {
			P_address = C_address;
			C_address = C_address + Size[C_opcode];
			P_opcode = C_opcode;
		}
		C_opcode = Program[C_address];
		// this is not precise
		C_operand = Program[C_address+1] + 256 * Program[C_address+2];

		if (Destination[C_address]) {
			X_val	= 0xffff;
			Y_val	= 0xffff;
			A_val	= 0xffff;
			continue;
		}

		if (Mode[C_opcode]) {
			switch (Register[C_opcode]) {
				case 'A':
					A_val	= 0xffff;
					break;
				case 'X':
					X_val	= 0xffff;
					break;
				case 'Y':
					Y_val	= 0xffff;
					break;
			}
			continue;
		}

		switch (C_opcode) {
			case ASM_JSR:
			case ASM_JMP:
			case ASM_RTS:
			case ASM_RTI:
				X_val = 0xffff; Y_val	= 0xffff; A_val	= 0xffff; break;
			case ASM_BNE:
			case ASM_BEQ:
			case ASM_BPL:
			case ASM_BMI:
				// this is not precise
				if (P_opcode == ASM_LDA ||
				    P_opcode == ASM_LDX ||
				    P_opcode == ASM_LDY)
					Keep[P_address] = true;
				break;
			// this is not precise
			case ASM_LDA:
				if (A_val == C_operand)
					Keep[C_address] = false;
				else
					A_val = C_operand;
				break;
			// this is not precise
			case ASM_LDX:
				if (X_val == C_operand)
					Keep[C_address] = false;
				else
					X_val = C_operand;
				break;
			// this is not precise
			case ASM_LDY:
				if (Y_val == C_operand)
					Keep[C_address] = false;
				else
					Y_val = C_operand;
				break;

			case ASM_TXA:
			case ASM_TYA:
			case ASM_PLA:
			//
			case ASM_ROL:
			case ASM_ROR:
			case ASM_ASL:
			case ASM_LSR:
			// these are not precise
			case ASM_ADC:
			case ASM_SBC:
			case ASM_EOR:
			case ASM_ORA:
			case ASM_AND:
				A_val = 0xffff; break;

			// these are not precise
			case ASM_STA:
				A_val = C_operand; break;
			case ASM_STX:
				X_val = C_operand; break;
			case ASM_STY:
				Y_val = C_operand; break;

			case ASM_TAX:
			case ASM_INX:
			case ASM_DEX:
			case ASM_TSX:
				X_val = 0xffff; break;
			case ASM_TAY:
			case ASM_INY:
			case ASM_DEY:
				Y_val = 0xffff; break;

			default:	// bit nop brk clv bvc bvs bcc bcs
					// clc sec cli sei sed cld txs php plp
				break;

		}
	}

	for (i=0;i<65536;i++)
		if (! Keep[i])
			reduction ++;
		
	if (reduction != 0) {
		printf("%d excess instructions need to be removed.\n", reduction);

/* code to compact program goes here
*/
	}

	return 0;
}

/* EOF
*/
