//
// NAME:
//   upl_comp.cpp
// TITLE:
//   UPL/Quetzalcoatl: Compiler.
// FUNCTION:
//   See header.
//
// AUTHOR:
//   Brendan Jones. (Contact through www.kdef.com/geek/vic)
// RIGHTS:
//   (c) Copyright Brendan Jones, 1998.  All Rights Reserved.
// SECURITY:
//   Unclassified.  
// LEGAL NOTICE:
//   See legal.txt before viewing, modifying or using this software.
// CONTACT:
//   Web:	http://www.kdef.com/geek/vic
//   Email:	See www.kdef.com/geek/vic
// DATE:
//   July 6, 1998.
// RIGHTS:
//  This file is part of The Quetzalcoatl Compiler.
//  
//  The Quetzalcoatl Compiler is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//  
//  The Quetzalcoatl Compiler is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//  
//  You should have received a copy of the GNU General Public License
//  along with The Quetzalcoatl Compiler; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
//
// MODIFICATIONS:
//   NAME  MOD  DATE       DESCRIPTION
//   bj    *    09oct2006  TODO: Array/Pointers aren't handled well here;
//			   They need a rewrite.
//
//
#ifdef DJGPP
#include <unistd.h>
#endif
#include "upl.h"
#include "implemen.h"


// TUNABLE DEFINE: NEW_CODE_MEMACCESS
//
// Harry uses NEW_CODE to mark code he added but isn't yet validated.
// His code to do memory/pointer access is faster than the old code
// and handles different types, so its NEW_CODE has been redefined
// as NEW_CODE_MEMACCESS and we'll try using it in the mainstream.
// [bj 09oct2006]
//
#define NEW_CODE_MEMACCESS  1



int	upl_Compiler::value_bytes(
		upl_value_type 	Value_type,
		boolean         Mandatory)

{
  switch (Value_type)
    {
    case upl_byte:
    case upl_char:
    case upl_boolean:
      return 1;

    case upl_ushort:
    case upl_short:
    case upl_pointer_byte:
    case upl_pointer_char:
    case upl_pointer_boolean:
    case upl_pointer_ushort:
    case upl_pointer_short:
      return 2;

    case upl_value_none:
    case upl_void:
    default:
      if (not Mandatory)
	return 0;

      abend(WHERE0, "Bad case");
    }

  #ifdef COMMON_NOREACHED
  return 0;
  #endif
}




// Normalise Binary.
//
// Operands require the arguments are of compatible types.
// For example, currently the UPLRTIME.ASM has no routine
// to multiply a word by a byte. So we must convert the 
// byte operand to a word, then we can use RUNTIME_MUL_WW.
//
// IN:
//   L		Lexical Stream.
//   C		Context.
//   Op		Binary operator (e.g. multiply).
//   Underneath	Type of argument underneath on the datastack.
//   Top	Type of argument on top of the datastack.
// OUT:
//   Unsigned	Set true iff this is an unsigned operation.
//   Parm_bytes	Size of largest result argument.
//
//
upl_value_type upl_Compiler::normalise_binary(
	Flex& 			L,
	upl_Context& 		C,
	upl_op			Op,
	upl_value_type		Underneath,
	upl_value_type		Top,
	boolean&		Unsigned,
	ushort		       *Parm_bytes)
{
  upl_value_type	result = Top,
			New_underneath,
			New_top;
  upl_Context_state	state;


  // Find out what the result of this operation will be,
  // and what the <New_top> and <New_underneath> value types are.
  // If these are different from the existing <Top> and <Underneath>,
  // then we must cast/convert them.
  //
  result = op_result(Op, Underneath, Top, New_underneath, New_top, Unsigned);


  /* COUT This is an example of how you *might* optimise binary byte-to-word casts
          from the Compiler Hacker's Guide. [bj 09oct2006]

  if ((Top == upl_byte		  || Top == upl_char		|| Top == upl_boolean)
  &&  (Underneath == upl_byte	  || Underneath == upl_char	|| Underneath == upl_boolean)
  &&  (New_top == upl_byte	  || New_top == upl_char	|| New_top == upl_boolean)
  &&  (New_underneath == upl_byte || New_underneath == upl_char || New_underneath == upl_boolean))
    {
    C.code.out(ASM_JSR);
    C.code.out_word_patch(RUNTIME_CAST_BB2WW);
    }
  else
    {
    ...
    */


  // Convert the top operand; we can do this as an unary (one argument) cast.
  //
  if (Top != New_top)
    {
    normalise_unary(L, C, Top, New_top);

    // Normalise_unary() returns true if it generates code for the conversion,
    // or false if it generates no code (because none is necessary).
    // In either case, we don't neeed to check this. All that is imporrant 
    // is that we have the desired binary value <New_top> on the top of the stack.
    }


  // The underneath operand is harder, because we already have the top argument in our way.
  // Solution is to generate code that pops the top register, stores it in <C>
  // (a pseudoregister defined in the runtime library), call normalise_unary()
  // to convert the underneath argument which is temporarily on the top,
  // then push the value in <C> back on top of the stack.
  //
  if (Underneath != New_underneath)
    {
    C.mark(state);  // Remember in case we want to rollback.

    // Write code to pop the top value into the <C> pseudoregister.
    //
    C.code.out(ASM_JSR);
    C.code.out_word_patch(
      value_bytes(New_top) == 1 ? RUNTIME_POP_B_C : RUNTIME_POP_W_C);

    // Now the <Underneath> value is temporarily on the top of the stack!
    // Call normalise_unary() to convert it.
    // It will return true if it generated code to convert it.
    // If will return false if the conversion turned out to be unnecessary.
    //
    if (normalise_unary(L, C, Underneath, New_underneath))
      {
      // Code was generated to do the conversion.
      // Now all we need do is push <C> back on the stack.
      //
      C.code.out(ASM_JSR);
      C.code.out_word_patch(
	value_bytes(New_top) == 1 ? RUNTIME_PUSH_B_C : RUNTIME_PUSH_W_C);
      }
    else
      //
      // Normalise unary did not generate any code.
      // This happens if the conversion is unnecessary.
      // For example, converting a ushort to a short.
      // If it didn't generate any code, then we can skip this 
      // part of the conversion.
      //
      C.rollback(state);
    }


  // If the passed us <Parm_bytes>, they want to know the size of the new arguments.
  // UNSURE: What if they have different sizes? At time of writing, operands
  //	     tend to assume both arguments are the same size, but what if in
  //	     the future we add operands this isn't true for? [bj 09oct2006]
  //
  if (Parm_bytes != NULL)
    *Parm_bytes = max(value_bytes(New_underneath), value_bytes(New_top));

  return result;
}




boolean upl_Compiler::normalise_unary(
	Flex& 			L,
	upl_Context& 		C,
	upl_value_type		Have,
	upl_value_type		Want)
{
  boolean	generated_code = false;


  if (Have != Want)
    if (Want == upl_void or Have == upl_void)
      L.parse_error("Cannot cast between a void and a sized data type.");
    else
      {
      if (Want == upl_ushort or Want == upl_short)
	switch (Have)
	  {
	  case upl_byte:
          case upl_boolean:	// HFD boolean is 00/ff
	    C.code.out(ASM_JSR);
	    C.code.out_word_patch(RUNTIME_CAST_B2W);
	    generated_code = true;
	    break;

	  case upl_char:
	    C.code.out(ASM_JSR);
	    C.code.out_word_patch(RUNTIME_CAST_C2W);
	    generated_code = true;
	    break;

	  case upl_ushort:
	  case upl_short:
	  case upl_pointer_byte:
	  case upl_pointer_char:
	  case upl_pointer_boolean:
	  case upl_pointer_ushort:
	  case upl_pointer_short:
	    break;

	  default:
	    abend(WHERE0, "Bad case");
	  }
      else if ((Want == upl_byte  or Want == upl_char or Want == upl_boolean)
	   and (Have == upl_short or Have == upl_ushort))
	  {
	  C.code.out(ASM_JSR);
	  C.code.out_word_patch(RUNTIME_CAST_W2B);
	  generated_code = true;
	  }
      }


  return generated_code;
}



/*
upl_value_type upl_Compiler::get_value_type(Flex& L)
{
  upl_value_type value_type;


  Select(L.matches("int")  or L.matches("short"))
      value_type = upl_short;
    when(L.matches("uint") or L.matches("ushort"))
      value_type = upl_ushort;

    when(L.matches("byte"))
      if (L.matches("*"))
	value_type = upl_pointer_byte;
      else
	value_type = upl_byte;

    when(L.matches("char"))
      value_type = upl_char;
    when(L.matches("boolean"))
      value_type = upl_boolean;
    when(L.matches("void"))
      value_type = upl_void;
    otherwise
      value_type = upl_value_none;
  endsel


  return value_type;
}
*/



void upl_Compiler::program(Flex& L, upl_Context& C, boolean Top_level)
{
  long			value;
  upl_symbol_token	token;
  boolean		defined_main = false;
  boolean		current_main;
  upl_Symbol	       *S;
  upl_Symbol const     *T;

  if (Top_level)
    C.predefine_types(L);




  loop
    if (L.matches("#"))
      {
      Select(L.matches("include"))
	  if (L.peek_type() == flex_string)
	    {
	    Flex I;

	    if (access(L.peek(), 0) == 0)
	      {
	      if (C.verbose >= 1)
		cerr << "I: Including file " << L.peek() << endl;

	      I.open_file((char *)L.get(NULL),
		  flex_syntax_c|flex_syntax_no_real|flex_syntax_lit_char);

	      program(I, C, false);

	      I.close();
	      }
	    else
	      L.parse_error("Could not open #include file");
	    }
	  else
	    L.parse_error("String expected; eg. #include \"header.h\"");

#ifndef NEW_CODE
	when(L.matches("define"))
	  upl_Expr_result	clause;

	  expr(L, C, clause, 0);

	  if (clause.constant)
	    S = C.symbols.declare(NULL, L, L.get_id(NULL),
	      clause.value_type, upl_constant, clause.value);
	  else
	    L.parse_error(
		"In this implementation #define must be assigned "
		"a constant numeric, boolean or character value.");
#else
	when(L.matches("define"))
	  {
	  upl_Expr_result	clause;

	  char token[UPL_TOKEN_LENGTH];
	  L.get_id(token);

	  expr(L, C, clause, 0);

	  if (clause.constant)
	    S = C.symbols.declare(NULL, L, token,
	      clause.value_type, upl_constant, clause.value);
	  else
	    L.parse_error(
		"In this implementation #define must be assigned "
		"a constant numeric, boolean or character value.");
	  }
#endif

	when(L.matches("pragma"))

	  Select(L.matches("page"))
	      L.check("align");

	      upl_Expr_result	clause;

	      expr(L, C, clause, 0);

	      if (clause.constant)
		{
		C.page_align_bytes = clause.value;
		C.know_alignment = true;
		}
	      else
		L.parse_error("Constant expression expected");

	      L.check("bytes");

	    otherwise
	      L.parse_error("Unknown pragma");
	  endsel


	otherwise
	  L.parse_error("Unrecognised preprocessor command.");

      endsel
      }
    else if (L.matches("cons"))
      {
      loop
	{
	L.get_id(token);
	L.check("=");
	value = L.get_int();

	C.symbols.declare(NULL, L, token, upl_byte, upl_constant, value, 0);

	if (not L.matches(","))
	  break;
	}

      L.check(";");
      }
    else if (L.matches("var"))
      {
      loop
	{
	C.symbols.declare(C.list_file, L, L.get_id(NULL), upl_byte, upl_variable,
	  C.data.out(0), 0);

	if (not L.matches(","))
	  break;
	}

      L.check(";");
      }
    else if (L.matches("typedef"))
      {
      T = type(L, C, true);

      S = C.symbols.declare(C.list_file,
	    L, L.get_id(NULL), T->value_type, upl_type, 0);

      L.check(";");
      }
    else
      {
      boolean	   constant  = false;
      boolean	   is_static = false;
#if 0
      boolean      data_is_constant = false;
#endif
#ifdef NO_WARNINGS
      upl_value_type value_type = upl_void;
      long	   quantity = 0;
      long	   value = 0;
#else
      upl_value_type value_type;
      long	   quantity;
      long	   value;
#endif

      loop
	Select(L.matches("const"))
	    constant    = true;
	  when(L.matches("static"))
	    is_static   = true;
#if 0
	  when(L.matches("fixed"))
	    data_is_constant   = true;
#endif
	  when(L.matches("auto"))
	    L.parse_error("\"auto\" variables may only be defined "
			  "in subroutines");
	  otherwise
	    break;
	endsel


      T = type(L, C, false);


      if (T != NULL)
	value_type = T->value_type;
      else
	if (constant or is_static)
	  L.parse_error("\"const\" or \"static\" must be followed by a type");
	else
	  break;


      boolean expect_semi = true;


      do
	{
	L.get_id(token);


	if (L.matches("["))
	  {
	  if (L.matches("]"))
	    quantity = -1;
	  else
	    {
	    upl_Expr_result	clause;

	    expr(L, C, clause, 0);

	    if (clause.constant)
	      quantity = clause.value;
	    else
	      L.parse_error("Constant expression expected");

	    L.check("]");

	    if (quantity < 1)
	      L.parse_error("Array dimension must be >= 1");
	    }
	  }
	else
	  quantity = 0;

	if (constant)
	  {
	  L.check("=");

	  upl_Expr_result	clause;

	  expr(L, C, clause, 0);

	  if (clause.constant)
	    value = clause.value;
	  else
	    L.parse_error("Constant expression expected");
	  }


	if (quantity == 0 and L.matches("("))
	  {
	  upl_value_type	parm_value_type;
	  long		parm_count = 0;
	  upl_addr  	local_addr = 0;
	  char		mangled_name[256];

	  upl_Symbol *D =
	  C.symbols.declare(C.list_file, L, token, value_type,
	    value_type == upl_void ? upl_procedure : upl_function,
	    C.code.current_addr(),
	    quantity);

	  strcpy(mangled_name, D->token);
	  strcat(mangled_name, "(");


	  if (not defined_main)
	    defined_main = current_main = equal(token, "main");
	  else
	    current_main = false;

	  if (L.matches("void"))
	    ;
	  else
	    {
	    while (notequal(L.peek(), ")"))
	      {
	      /*
	      const upl_Symbol *parm_type =
		  C.symbols.get(L, L.get_id(NULL), true);

	      parm_value_type = parm_type->value_type;
	      */
	      const upl_Symbol *parm_type = type(L, C, false);
	      parm_value_type = parm_type->value_type;


	      Select(parm_value_type == upl_value_none)
		  break;

		when(parm_value_type == upl_void)
		  L.parse_error("\"void\" is not a valid parameter type");

		otherwise
		  D->parms.size(parm_count+1);
		  D->parms[parm_count++] = parm_value_type;
	      endsel

	      if (L.peek_type() != flex_id)
		L.parse_error("identifier expected for parameter name");

	      upl_Symbol *Parm = C.symbols.declare(C.list_file,
		  L,
		  L.peek(),
		  parm_value_type,
		  upl_variable,
		  local_addr,
		  0,
		  true,
		  0,
		  parm_type);
	      Parm->in_local_storage = true;

	      append_mangling(mangled_name, parm_value_type);

	      L.get(NULL);

	      local_addr += parm_type->size_bytes;

	      if (not L.matches(","))
		break;
	      }
	    }

	  L.check(")");
	  strcat(mangled_name, ")");

	  // Append the result type.
	  //
	  if (D->value_type != upl_void)
	    append_mangling(mangled_name, D->value_type);


	  if (L.matches(";"))
	    {
	    // External reference here.
	    //
	    D->external_id = C.get_next_external_id();

	    // Add an external application name.
	    //
	    C.code.add_external_name(L, mangled_name, D->external_id);
	    }
	  else
	    {
	    // Body declared here.
	    //

	    // Remember if we must do the BP prolog/epilog.
	    //
	    // We store the number of bytes allocated locally here.
	    //
	    D->is_local = local_addr;


	    if (not is_static)
	      C.code.add_name(L, mangled_name, C.code.current_addr());


	    if (current_main)
	      {
	      if (D->parms.size() != 0 or value_type != upl_void)
		L.parse_error("void main(void) is the only allowable definition of main");

	      C.code.out(ASM_JSR);
	      C.code.out_word_patch(RUNTIME_INIT);
	      }


	    if (D->size_bytes > 2)
	      L.parse_error("You cannot pass this type of value "
			    "as the result of a subroutine.");


	    boolean unreachable_code = false;

	    C.subroutine_begin(D->value_type, D->is_local == 0);
	    C.subroutine_local_addr = local_addr;
	    C.local_parm_bytes	    = local_addr;
	    compound_statement(L, C, &unreachable_code, true);
	    C.symbols.reset_locals();


	    // Hack off any final jump to where we are now.
	    //
	    if (C.return_patches >= 1)
	      if (C.return_patch[C.return_patches-1] ==
		  C.code.current_addr() - 2)
		{
		C.code.shorten(3);	// Skip that final jump.
		C.return_patches--;
                }


	    C.subroutine_end(L, C.code.current_addr());

	    /*
	    if (unreachable_code and D->size_bytes > 0)
	      L.parse_error("Function is missing a return statement");
	     */


	    // Discard local storage; parameters and local variables.
	    //
	    if (D->is_local > 0)
	      {
	      // Save the register results which EPILOG will trash.
	      // Actually Epilog trashes only <A> and <Y>;
	      // <X> is safe.
	      //
	      if (D->size_bytes > 0)
		switch (D->size_bytes)
		  {
		  case 2:
		    C.code.out(ASM_PHA);
		    break;

		  case 1:
		    C.code.out(ASM_TAX);

		  case 0:
		    break;

		  default:
		    abend(WHERE0, "Bad case");
		  }


	      C.code.out(ASM_LDA_IMM);

	      C.code.out(C.max_subroutine_local_addr);

	      if (D->size_bytes == 0)
		C.code.out(ASM_JMP);
	      else
		C.code.out(ASM_JSR);

	      C.code.out_word_patch(RUNTIME_EPILOG);


	      if (D->size_bytes > 0)
		{
		switch (D->size_bytes)
		  {
		  case 2:
		    C.code.out(ASM_PLA);
		    break;

		  case 1:
		    C.code.out(ASM_TXA);
		    break;

		  case 0:
		    break;


		  default:
		    abend(WHERE0, "Bad case");
		  }

		C.code.out(ASM_RTS);
		}
	      }
	    else if (not unreachable_code)
	      C.code.out(ASM_RTS);


	    C.symbols.reset_locals();
	    }


	  expect_semi = false;

	  break;
	  }
	else
	  {
	  boolean	  initialised  	= false;
	  upl_addr  start_addr   	= C.data.current_addr();
#ifdef NO_WARNINGS
	  long	  datum = 0;
#else
	  long	  datum;
#endif
	  long	  datums	= 0;
	  boolean	  array		= quantity > 0;  // ?
	  upl_Expr_result	clause;


	  if (not constant)
	    if (quantity < 0)
	      {
	      L.check("=");
	      initialised = true;
	      }
	    else
	      initialised = L.matches("=");


	  if (initialised)
	    {
	    datums = 0;

	    Select(L.peek_type() == flex_string)
		if (value_type == upl_char or value_type == upl_byte)
		  {
		  datums = strlen(L.peek())+1;
		  C.data.out_string(L.get_string(), C);
		  }
		else
		  L.parse_error("Can only initial char or byte arrays with strings");

	      otherwise
		 if (array)
		   L.check("{");

		loop
		  {
		  if (array and L.matches("}"))
		    break;


		  expr(L, C, clause, 0);

		  if (clause.constant)
		    datum = clause.value;
		  else
		    L.parse_error("Constant expression expected");


		  datums++;

		  switch (value_type)
		    {
		    case upl_byte:
		    case upl_char:
		    case upl_boolean:
		      C.data.out(datum);
		      break;

		    case upl_short:
		    case upl_ushort:
		      C.data.out_word(datum);
		      break;

		    case upl_pointer_byte:
		    case upl_pointer_char:
		    case upl_pointer_boolean:
		    case upl_pointer_ushort:
		    case upl_pointer_short:
			C.data.out_addr_data(datum);
		      //C.data.out_word(datum);
		      /*
		      L.parse_error("In this implementation you cannot "
				    "initialise a pointer");

		       */
		      break;

		    default:
		      L.parse_error("Cannot initialise this variable type");
		    }

		  if (not array)
		    break;
		  else if (not L.matches(","))
		    {
		    L.check("}");
		    break;
		    }
		  }
	    endsel
	    }


	  if (quantity < 0)
	    quantity = datums;
	  else if (quantity > 0 and datums > quantity)
	    L.parse_error("Initialisation data exceeds array bounds");


	  S = C.symbols.declare(C.list_file, L, token, value_type,
	    constant ? upl_constant : upl_variable,
	    constant ? value 	  : start_addr,
	    quantity, false, 0, T);


	  if (not constant)
	    {
	    quantity = max(quantity, 1);

	    for (long c=datums; c<quantity; c++)
	      switch (S->value_bytes)
		{
		case 1:  C.data.out(0);	   break;
		case 2:  C.data.out_word(0); break;

		default:
		  for (long b=0; b<S->value_bytes; b++)
		    C.data.out(0);
		  break;
		}


	    // If it's an array see if we can do fast access.
	    //
	    // The whole array must be within the same page.
	    //
	    S->fast =
	      quantity > 0 and S->size_bytes <= 256 and C.know_alignment and
	       (((S->value + C.page_align_bytes) >> 8) ==
		((S->value + C.page_align_bytes + S->size_bytes-1) >> 8));
	    }
	  }
	}
      while (L.matches(","));


      if (expect_semi)
	L.check(";");
      }


  if (not defined_main and (
     equal(L.peek(), "{") or equal(L.peek(), "[")))
    {
    boolean c_style = equal(L.peek(), "{");

    C.code.add_name(L, "main()", C.code.current_addr());

    C.code.out(ASM_JSR);
    C.code.out_word_patch(RUNTIME_INIT);

    boolean unreachable_code = false;

    C.subroutine_begin(upl_value_none, true);
    C.subroutine_local_addr = 0;
    C.local_parm_bytes	    = 0;
    compound_statement(L, C, &unreachable_code);
    C.subroutine_end(L, C.code.current_addr());

    if (not unreachable_code)
      C.code.out(ASM_RTS);

    if (not c_style)
      L.check(".");
    }

  if (not L.eof())
    L.parse_error("Type declaration or preprocessor command expected");



  if (Top_level)
    if (C.know_alignment and C.needed_alignment)
      {
      C.data.page_aligned 	= true;
      C.data.absolute_aligned 	= false;
      C.data.alignment_addr	= C.page_align_bytes;
      }
}




boolean upl_Compiler::statement(
	Flex& 		 L,
	upl_Context& 	 C,
	boolean 	*Unreachable_code,
	boolean		 Semicheck)
{
  const char   *P 	    = L.peek();
  boolean	check_semi  = true;


  Select(equal(P, "{") or equal(P, "["))
      compound_statement(L, C);
      check_semi = false;

#if 0
    when(equal(P, ";"))
      check_semi = false;
#endif

    when(equal(P, "if"))
      if_statement(L, C);
      check_semi = false;


    when(equal(P, "while"))
      while_statement(L, C);
      check_semi = false;


    when(equal(P, "repeat") or equal(P, "do"))
      repeat_statement(L, C);


    when(equal(P, "for"))
      for_statement(L, C);
      check_semi = false;


    when(equal(P, "call"))
      call_statement(L, C);


    when(equal(P, "return"))
      return_statement(L, C);
      if (Unreachable_code)
	*Unreachable_code = true;


    when(equal(P, "put")  or equal(P, "putln") or
	 equal(P, "cout") or equal(P, "cerr"))
      put_statement(L, C);


    when(equal(P, "inc") or equal(P, "dec") or
	 equal(P, "++")  or equal(P, "--"))
      inc_statement(L, C, true, NULL, NULL);


    when(equal(P, "memory") or equal(P, "mem") or equal(P, "*"))
      assign_memory_statement(L, C);


    when(equal(P, "ioctl"))
      ioctl_statement(L, C);


    when(L.peek_type() == flex_id)
      const upl_Symbol *D = C.symbols.get(L, L.peek(), true);

      switch (D->symbol_type)
	{
	case upl_variable:
	  L.get(NULL);
	  assign_statement(L, C, *D);
	  break;

	case upl_procedure:
	case upl_function:
	  L.get(NULL);
	  subroutine_statement(L, C, *D, NULL);
	  break;

	case upl_constant:
	  L.parse_error("A statement may not begin with a constant.");

	case upl_symbol_none:
	default:
	  L.parse_error("Statement expected");
	  //abend(WHERE0, "Bad case");
	}

    otherwise
      return false;
  endsel


  if (check_semi and Semicheck)
    L.check(";");


  return true;
}




void upl_Compiler::compound_statement(
	Flex& 		 L,
	upl_Context& 	 C,
	boolean 	*Unreachable_code,
	boolean		 Initialise_auto)
{
  boolean	c_style = L.matches("{");
  long		locals	= C.symbols.mark_locals();
  long		local_addr = C.subroutine_local_addr;

  if (not c_style)
    L.check("[");
  else
    loop
      {
      boolean	is_static;


      Select(L.matches("static"))
	  is_static = true;


	when(L.matches("auto"))
	  is_static = false;

	  if (not C.has_auto_variables and not Initialise_auto)
	    L.parse_error(
		"You cannot declare an \"auto\" variable in an inner compound "
		"statement without any \"auto\" variables in the top-most "
		"compound statement of the subroutine.");

	otherwise
	  break;
      endsel


      upl_Symbol const *T = type(L, C, true);
      upl_Symbol       *S;
      upl_addr	variable_address;


      do
	{
	if (L.peek_type() != flex_id)
	  L.parse_error("Identifier expected");


	if (is_static)
	  {
	  variable_address = C.data.current_addr();

	  switch (T->value_bytes)
	    {
	    case 1:  C.data.out(0);	break;
	    case 2:  C.data.out_word(0);  break;

	    default:
	      for (long b=0; b<T->value_bytes; b++)
		C.data.out(0);
	      break;
	    }
	  }
	else
	  {
	  variable_address = C.subroutine_local_addr;

	  C.subroutine_local_addr += T->value_bytes;
	  }


	S = C.symbols.declare(C.list_file, L, L.peek(), T->value_type,
		upl_variable, variable_address,  0, true, 0, T);

	if (not is_static)
	  S->in_local_storage = true;

	L.get(NULL);
	}
      while (L.matches(","));

      L.check(";");
      }


  if (Initialise_auto)
    if (local_addr < C.subroutine_local_addr)
      {
      C.code.out(ASM_LDA_IMM);

      C.auto_variable_addr = C.code.current_addr();
      C.code.out(0);

      C.code.out(ASM_JSR);
      C.code.out_word_patch(RUNTIME_PUSH_N);

      C.has_auto_variables = true;
      }

#ifndef GCC
  while (not L.matches(c_style ? "}" : "]"))
#else
  while (not L.matches(c_style ? (char *)"}" : (char *)"]"))
#endif
    {
    if (not statement(L, C, Unreachable_code))
      break;

    /*
    if (c_style)
      L.check(";");
    else
      if (not L.matches(";"))
	break;
      */
    }

  C.max_subroutine_local_addr =
	max(C.max_subroutine_local_addr,
	C.subroutine_local_addr);


  C.subroutine_local_addr = local_addr;


  C.symbols.release_locals(locals);
}



// Generates code to skip over a 3 byte JMP statement
// iff the condition <Clause> is true.
//
//
void upl_Compiler::condition_flow(
	Flex&			L,
	upl_Context& 		C,
	upl_Expr_result&        Clause,
	boolean			Skip_if_true)
{
  boolean	compared = false;


  Select(Clause.constant)
      L.parse_warning(Clause.value
	     ? "Conditional expression is always true"
	     : "Conditional expression is always false");

      Clause.set_constant(L, Clause.value != 0);

      C.code.out(ASM_LDA_IMM);
      C.code.out(Clause.constant);


    when(Clause.transient == upl_transient_znc)
      C.rollback(Clause.transient_state);
      compared = true;


    otherwise
      load_reg(C, Clause);
  endsel


  // Set up the comparison flags.
  //
  if (not compared)
    switch (Clause.value_bytes)
      {
      case 2:
	C.code.out(ASM_CMP_IMM);
	C.code.out(0);
	C.code.out(ASM_BNE);
	C.code.out(2);
	C.code.out(ASM_CPX_IMM);
	C.code.out(0);
	break;

      case 1:
	C.code.out(ASM_CMP_IMM);
	C.code.out(0);
	break;

      default:
	abend(WHERE0, "Bad case");
      }


  if (Clause.transient == upl_transient_znc)
    switch (Clause.transient_op)
      {
      case upl_eq:
	C.code.out(Skip_if_true ? ASM_BEQ : ASM_BNE);	// If false, skip over the else jump
	C.code.out(3);
	break;

      case upl_ne:
	C.code.out(Skip_if_true ? ASM_BNE : ASM_BEQ);  // If false, skip over the else jump
	C.code.out(3);
	break;

      default:
	abend(WHERE0, "Bad case");
      }
  else
    {
    C.code.out(Skip_if_true ? ASM_BNE : ASM_BEQ);  // If false, skip else jump
    C.code.out(3);
    }
}




void upl_Compiler::if_statement(Flex& L, upl_Context& C)
{
  upl_Expr_result	clause;
  upl_addr 		to_else,
			to_end;


  L.check("if");

  if (L.matches("("))
    {
    expr(L, C, clause, 0);
    L.check(")");
    L.matches("then");
    }
  else
    {
    expr(L, C, clause, 0);
    L.check("then");
    }


  condition_flow(L, C, clause, true);

  C.code.out(ASM_JMP);
  to_else = C.code.current_addr();
  C.code.out_word(0, upl_code_word);

  statement(L, C);


  if (L.matches("else"))
    {
    C.code.out(ASM_JMP);
    to_end = C.code.current_addr();
    C.code.out_word(0, upl_code_word);

    C.code.set_word(to_else, C.code.current_addr(), upl_code_word);

    statement(L, C);

    C.code.set_word(to_end,  C.code.current_addr(), upl_code_word);
    }
  else
    C.code.set_word(to_else, C.code.current_addr(), upl_code_word);
}




void upl_Compiler::while_statement(Flex& L, upl_Context& C)
{
  upl_Expr_result	clause;
  upl_addr 		condition_label = C.code.current_addr(),
			to_end;


  L.check("while");
  if (L.matches("("))
    {
    expr(L, C, clause, 0);
    L.check(")");
    L.matches("do");
    }
  else
    {
    expr(L, C, clause, 0);
    L.check("do");
    }

  condition_flow(L, C, clause, true);

  C.code.out(ASM_JMP);
  to_end = C.code.current_addr();
  C.code.out_word(0, upl_code_word);

  statement(L, C);

  C.code.out(ASM_JMP);
  C.code.out_word(condition_label, upl_code_word);

  C.code.set_word(to_end, C.code.current_addr(), upl_code_word);
}




void upl_Compiler::repeat_statement(Flex& L, upl_Context& C)
{
  upl_Expr_result	clause;
  upl_addr 		loop_label = C.code.current_addr();
  boolean		loop_if_true;

  boolean c_style = L.matches("do");
  if (not c_style)
    L.check("repeat");

  loop
    {
    statement(L, C);

    if (c_style)
      {
      if (equal(L.peek(), "while"))
	break;
      }
    else
      if (not L.matches(";"))
	break;
    }

  if (c_style)
    {
    L.check("while");
    expr(L, C, clause, 0);
    loop_if_true  = true;
    }
  else
    {
    L.check("until");
    expr(L, C, clause, 0);
    loop_if_true  = false;
    }

  condition_flow(L, C, clause, not loop_if_true);

  C.code.out(ASM_JMP);
  C.code.out_word(loop_label, upl_code_word);
}




void upl_Compiler::for_statement(Flex& L, upl_Context& C)
{
  upl_Expr_result	clause;
  upl_addr 		test_label,
			inc_label,
			to_body,
			to_end;

  L.check("for");
  L.check("(");

  do
    statement(L, C, NULL, false);
  while (L.matches(","));

  L.check(";");

  test_label = C.code.current_addr();
  expr(L, C, clause, 0);
  condition_flow(L, C, clause, false);

  C.code.out(ASM_JMP);
  to_body    = C.code.current_addr();
  C.code.out_word(test_label, upl_code_word);

  C.code.out(ASM_JMP);
  to_end     = C.code.current_addr();
  C.code.out_word(test_label, upl_code_word);

  L.check(";");
  inc_label = C.code.current_addr();

  do
    statement(L, C, NULL, false);
  while (L.matches(","));

  C.code.out(ASM_JMP);
  C.code.out_word(test_label, upl_code_word);
  L.check(")");

  C.code.set_word(to_body, C.code.current_addr(), upl_code_word);

  statement(L, C);

  C.code.out(ASM_JMP);
  C.code.out_word(inc_label, upl_code_word);

  C.code.set_word(to_end, C.code.current_addr(), upl_code_word);
}




void upl_Subscript::read(
	Flex& 			L,
	upl_Context& 		C,
	const upl_Symbol& 	S)
{
  upl_Context_state	state;

  C.mark(state);

  use_fast_variable	=
  use_constant		=
  use_stack		= false;


  L.check("[");
  upl_Compiler::expr(L, C, index, 0);
  L.check("]");


  Select(index.constant)
      use_constant	= true;

    when(index.is_variable()
     and index.variable->size_bytes == 1
     and not index.variable->in_local_storage
     and S.fast)
	use_fast_variable = true;
	C.rollback(state);

    otherwise
      use_stack = true;

      upl_Compiler::normalise_unary(L, C, index.value_type, upl_ushort);
  endsel
}





  /*
void upl_Compiler::variable(
	Flex& 			L,
	upl_Context& 		C,
	const upl_Symbol& 	S)
{

  while (S->type)
    {
    }

  Select(L.matches("["))

}
  */




void upl_Compiler::assign_statement(
	Flex& 			L,
	upl_Context& 		C,
	const upl_Symbol& 	S)
{
  upl_Expr_result	clause;
  upl_Subscript         subscript;
  boolean		subscripted;


  subscripted = S.count != 0;

  if (is_pointer(S.value_type) and equal(L.peek(), "["))
    subscripted = true;

  if (subscripted)
    subscript.read(L, C, S);

  /*
  if (S.fields.size() > 0)
    {
    L.check(".");

    boolean	found = false;

    for (int field_i=0; field_i<fields.size(); field_i++)
      if (equal(S.field[field_i], L.peek())
    }
    */



  Select (L.matches("="))
      expr(L, C, clause, 0);

      Select(clause.constant)
	  reg_var(C, S, subscripted ? &subscript : NULL, false,
	    true, clause.value, 0, is_pointer(S.value_type));

	otherwise
	  load_reg(C, clause);
	  reg_var(C, S, subscripted ? &subscript : NULL, false,
	    false, 0, 0, is_pointer(S.value_type));
      endsel

    when(equal(L.peek(), "++") or equal(L.peek(), "--"))
      inc_statement(L, C, false, &S, subscripted ? &subscript : NULL);

    otherwise
      L.parse_error("\"=\", \"++\" or \"--\" expected");
  endsel
}




void upl_Compiler::ioctl_statement(
	Flex& 			L,
	upl_Context& 		C)
{

#ifdef NO_WARNINGS
  upl_Expr_result	clause_mode;
#else
  upl_Expr_result	clause_handle, clause_mode;
#endif

  L.check("ioctl");
  L.check("(");
  if (not L.matches("stdin"))
    L.check("cin");
  L.check(",");
  L.check("ioctl_set");
  L.check(",");
  expr(L, C, clause_mode, 0);
  ensure_pushed(L, C, clause_mode);
  L.check(")");

  normalise_unary(L, C, clause_mode.value_type, upl_byte);
  C.code.out(ASM_JSR);
  C.code.out_word_patch(RUNTIME_IOCTL);
}




void upl_Compiler::assign_memory_statement(
	Flex& 			L,
	upl_Context& 		C)
{

#ifndef NO_WARNINGS
  upl_Context_state	key_state;
#endif
  upl_Expr_result	clause_addr, clause_content;
  boolean		pointer_assignment = L.matches("*");
  upl_value_type 	vtype	= upl_byte;
  int	       		vbytes	= 1;


  if (not pointer_assignment)
    {
    if (not L.matches("mem"))
      L.check("memory");

    if (L.matches("."))
      {
      const upl_Symbol *T = type(L, C, true);
      vtype  = T->value_type;
      vbytes = T->value_bytes;
      }

    L.check("[");
    }

  expr(L, C, clause_addr, 0);

  if (not clause_addr.constant)
    {
    ensure_pushed(L, C, clause_addr);
    normalise_unary(L, C, clause_addr.value_type, upl_ushort);

    if (pointer_assignment)
      {
      vtype  = pointer_value_type(clause_addr.value_type);
      vbytes = value_bytes(vtype);
      }
    }


  if (not pointer_assignment)
    {
    L.check("]");
    }

  L.check("=");

  expr(L, C, clause_content, 0);


  Select(clause_addr.constant)
      //
      // Could use load_reg() for both, but this saves
      // us unnecessarily loading the X reg.
      //
      if (clause_content.constant)
	{
	load_reg(C, clause_content);
	//C.code.out(ASM_LDA_IMM);	-- load reg already does this.
	//C.code.out(clause_content.value);
	}
      else
	load_reg(C, clause_content);


      if (-128 <= clause_addr.value and clause_addr.value <= 255)
	{
	C.code.out(ASM_STA_Z);
	C.code.out(clause_addr.value);
#ifdef NEW_CODE
	// if mem.int[]
	if (vbytes == 2) {			//HFD this works
		C.code.out(ASM_STX_Z);
		C.code.out(clause_addr.value + 1);
		}
#endif
	}
      else
	{
	C.code.out(ASM_STA);
	C.code.out_word(clause_addr.value);
	// if mem.int[]
#ifdef NEW_CODE
	if (vbytes == 2) {			//HFD this works
		C.code.out(ASM_STX);
		C.code.out_word(clause_addr.value + 1);
		}
#endif
	}


    otherwise
      ensure_pushed(L, C, clause_content);
      normalise_unary(L, C, clause_content.value_type, vtype);


      C.code.out(ASM_JSR);
      C.code.out_word_patch(
		vbytes == 1 ? RUNTIME_POKE_W_B : RUNTIME_POKE_W_W);

  endsel
/* HFD  - fix required for this...

  *pointer = &variable;		converts to
	lda #pointer-lo
	ldx #pointer-hi		instead of
	lda  pointer-lo
	ldx  pointer-hi

	vbytes is always 2 for pointers
*/
}



void upl_Compiler::put_statement(
	Flex& 		L,
	upl_Context& 	C)
{
  boolean	    stream = false;
  boolean 	    newline = false;
  upl_Expr_result   clause;
  upl_addr	    string_addr;


  Select(L.matches("put"))
      ;

    when(L.matches("putln"))
      newline = true;

    when(L.matches("cout") or L.matches("cerr"))
      stream = true;

    otherwise
      L.parse_error("\"put\" or \"putln\" expected");
  endsel


  if (not stream and L.matches("(") or
	  stream and L.matches("<<"))
    {
    do
      {
      if (L.peek_type() == flex_string)
	{
	string_addr = C.data.out_string(L.get_string(), C);

	C.code.out(ASM_LDA_IMM);
	C.code.out_word((string_addr),    upl_data_byte_lo);

	C.code.out(ASM_LDX_IMM);
//	C.code.out_word((string_addr>>8), upl_data_byte_hi);	// Is MSB part now done automatically? [bj 21sep98]
	C.code.out_word((string_addr), upl_data_byte_hi);

	C.code.out(ASM_JSR);
	C.code.out_word_patch(RUNTIME_PRINT_STRING_AX);
	}
      else
	{
	expr(L, C, clause, 0);

	if (clause.is_constant() and
	    clause.value_type == upl_char and
	    clause.value      == '\n')
	  {
	  C.code.out(ASM_JSR);
	  C.code.out_word_patch(RUNTIME_PRINT_NL);
	  }
	else if (clause.is_variable()
	    and clause.value_type 	    == upl_char
	    and clause.variable->count      != 0)
	  {
	  load_reg(C, clause);

	  C.code.out(ASM_JSR);
	  C.code.out_word_patch(RUNTIME_PRINT_STRING_AX);
	  }
	else
	  {
	  ushort subroutine = 0;

	  load_reg(C, clause);

	  switch (clause.value_bytes)
	    {
	    case 1:
	      if (clause.value_type == upl_char)
		{
		subroutine = RUNTIME_PRINT_CH_A;
		break;
		}
	      else
		{
		if (clause.value_type == upl_boolean)
		  {
		  C.code.out(ASM_JSR);
		  C.code.out_word_patch(RUNTIME_EXTEND_AX);
		  }
		else
		  {
		  C.code.out(ASM_LDX_IMM);
		  C.code.out(0);
		  }
		/*FALLTHROUGH*/;
		}

	    case 2:
	      switch (clause.value_type)
		{
		case upl_short:
		case upl_boolean:
		  subroutine = RUNTIME_PRINT_INTEGER_AX_SIGNED;
		  break;

		case upl_pointer_byte:
		case upl_pointer_char:
		  subroutine = RUNTIME_PRINT_STRING_AX;
		  break;

		case upl_pointer_boolean:
		case upl_pointer_ushort:
		case upl_pointer_short:
		  L.parse_error(
			"In this implementation you cannot print this type. "
			"Cast it to an integer if you want to see the address.");
		  break;

		default:
		  subroutine = RUNTIME_PRINT_INTEGER_AX;
		}
	      break;

	    default:
	      L.parse_error("In this implementation you cannot "
			    "print this type");
	    }

	  C.code.out(ASM_JSR);
	  C.code.out_word_patch(subroutine);
	  }
	}
      }
    while (not stream and L.matches(",") or
	       stream and L.matches("<<"));


    if (not stream)
      L.check(")");
    }
  else if (not newline)
#ifndef GCC
    L.check(stream ? "<<" : "(");
#else
    L.check(stream ? (char *)"<<" : (char *)"(");
#endif
  

  if (newline)
    {
    C.code.out(ASM_JSR);
    C.code.out_word_patch(RUNTIME_PRINT_NL);
    }
}




void upl_Compiler::ensure_pushed(
	Flex& 		  L,
	upl_Context& 	  C,
	upl_Expr_result&  Result)
{
  ushort 	    size_bytes = 0;
  upl_Context_state state;


  if  (not Result.is_pushed)
    {
    Result.is_pushed = true;

    Select(Result.is_variable())
	reg_var(C, *Result.variable, NULL, true, false);
	size_bytes = Result.variable->size_bytes;


      when(Result.is_constant())
	boolean epointer = is_pointer(Result.value_type);

	switch (Result.value_bytes)
	  {
	  case 2:
	    C.code.out(ASM_LDX_IMM);
	    if (epointer)
	      C.code.out_word(Result.value, upl_data_byte_hi);
	    else
	      C.code.out(Result.value>>8);

	    size_bytes++;
	    /*FALLTHROUGH*/

	  case 1:
	    size_bytes++;
	    C.code.out(ASM_LDA_IMM);
	    if (epointer)
	      C.code.out_word(Result.value, upl_data_byte_lo);
	    else
	      C.code.out(Result.value);
	    break;

	  default:
	    abend(WHERE0, "Bad case");
	  }

      otherwise
	return;
    endsel

#ifdef NO_WARNINGS
    ushort runtime_push = 0;
#else
    ushort runtime_push;
#endif

    switch (size_bytes)
      {
      case 1: runtime_push = RUNTIME_PUSH_B; break;
      case 2: runtime_push = RUNTIME_PUSH_W; break;
      default:
	L.parse_error(
	  "Cannot use this type of variable directly in an expression");
      }

    C.mark(state);

    Result.set_transient_only(
      size_bytes == 1 ? upl_transient_a : upl_transient_ax, &state);

    C.code.out(ASM_JSR);
    C.code.out_word_patch(runtime_push);
    }
}




void upl_Compiler::subroutine_statement(
	Flex& 			L,
	upl_Context& 		C,
	const upl_Symbol& 	S,
	upl_Expr_result	       *Result)
{
//const upl_Symbol      S;
  upl_Expr_result	clause;


  /*
  if (S.symbol_type != upl_procedure )
    L.parse_error("procedure name expected");
    */


  // If uses the BP...
  //
#ifdef NEW_CODE
  if (S.in_local_storage or (S.parms.items > 0))
#else
  if (S.in_local_storage)
#endif
    {
    C.code.out(ASM_JSR);
    C.code.out_word_patch(RUNTIME_PROLOG);
    }

  if (S.parms.items > 0)
    {
    L.check("(");
    for (int p=0; p<S.parms.size(); p++)
      {
#ifndef NEW_CODE
      if (equal(L.peek(), ")"))
	;
      else
	L.parse_error("Missing parameters in subroutine call");
#else
      if (equal(L.peek(), ")"))
	L.parse_error("Missing parameters in subroutine call");
#endif

// ------------------------
// HFD would be nice to verify no auto variables
//     used anywhere as a parameter
// ------------------------

      expr(L, C, clause, 0);

      ensure_pushed(L, C, clause);

      normalise_unary(L, C, clause.value_type, S.parms[p]);

      if (p != S.parms.items-1)
	L.check(",");
      }
    L.check(")");
    }
  else if (L.matches("("))
    L.check(")");


  C.code.out(ASM_JSR);

  if (S.external_id)
    C.code.out_word_patch(S.external_id);
  else
    C.code.out_word(S.value, upl_code_word);


  if (Result != NULL)
    {
    upl_Context_state key_state;

    C.mark(key_state);

    switch (S.size_bytes)
      {
      case 1:
	Result->set_value_type(S.value_type, 0, upl_transient_a, &key_state);
	C.code.out(ASM_JSR);
	C.code.out_word_patch(RUNTIME_PUSH_B);
	break;

      case 2:
	Result->set_value_type(S.value_type, 0, upl_transient_ax, &key_state);
	C.code.out(ASM_JSR);
	C.code.out_word_patch(RUNTIME_PUSH_W);
	break;

      default:
	abend(WHERE0, "Bad case");
      }
    }
}




void upl_Compiler::return_statement(
	Flex& 			L,
	upl_Context& 		C)
{
  upl_Expr_result	clause;
  upl_Context_state	key_state;


  if (not C.in_subroutine)
    L.parse_error("You may only return from within a subroutine");


  L.check("return");


  if (C.return_value != upl_value_none)
    {
     expr(L, C, clause, 0);

     C.mark(key_state);

     ensure_pushed(L, C, clause);
     if (not normalise_unary(L, C, clause.value_type, C.return_value))
       C.rollback(key_state);

     load_reg(C, clause);
     }


  if (C.return_immediate)
    C.code.out(ASM_RTS);
  else
    {
    C.code.out(ASM_JMP);
    C.subroutine_return(C.code.current_addr());
    C.code.out_word(0, upl_code_word);
    }
}






// A UPL "call" statement.
//
// 'C' does not use this.
//
void upl_Compiler::call_statement(
	Flex& 			L,
	upl_Context& 		C)
{
  upl_Expr_result	clause;
  boolean		with_reg = false;
  const upl_Symbol     *reg[4]    = {NULL, NULL, NULL, NULL};
  const char 	       *reg_name[] = {"p", "a", "x", "y"};
  int			regs = 0;
  int 			reg_i;
  boolean		reg_in = false, reg_out = false;


  L.check("call");
  expr(L, C, clause, 0);
  if (not clause.constant)
    L.parse_error("call may only be used with a constant value");

  if (L.matches("with"))
    {
    if (not L.matches("reg"))
      L.check("registers");

    loop
      Select(L.matches("in"))	  reg_in = true;
	when(L.matches("out"))	  reg_out = true;
	otherwise
	  break;
      endsel

    if (not reg_in and not reg_out)
      reg_in = reg_out = false;


    with_reg = true;
    }


  if (with_reg)
    {
    for (reg_i=0; reg_i<4; reg_i++)
      {
      reg[reg_i] = C.symbols.get(L, reg_name[reg_i], false);

//HFD this is much easier
      if (reg[reg_i])
	if (reg[reg_i]->count != 0)
		L.parse_error("Cannot have an array named a x y p and use call");

//HFD this was much too complicated
      if (reg_in)
        if (reg[reg_i])
	  regs++;
      }

    if ((regs != 0) and reg_in) {
	if (reg[3]) {
		C.code.out(ASM_LDY);
		C.code.out_addr_data(reg[3]->value);
		}
	if (reg[2]) {
		C.code.out(ASM_LDX);
		C.code.out_addr_data(reg[2]->value);
		}
	if (reg[0]) {
		C.code.out(ASM_LDA);
		C.code.out_addr_data(reg[0]->value);
		C.code.out(ASM_PHA);
		}
	if (reg[1]) {
		C.code.out(ASM_LDA);
		C.code.out_addr_data(reg[1]->value);
		}
	if (reg[0])
		C.code.out(ASM_PLP);
	}
    }

  C.code.out(ASM_JSR);
  C.code.out_word(clause.value);

//HFD this was much too complicated as well
  if (with_reg and reg_out)
    if ((regs != 0) and reg_out) {
	if (reg[0])
		C.code.out(ASM_PHP);
	if (reg[3]) {
		C.code.out(ASM_STY);
		C.code.out_addr_data(reg[3]->value);
		}
	if (reg[2]) {
		C.code.out(ASM_STX);
		C.code.out_addr_data(reg[2]->value);
		}
	if (reg[1]) {
		C.code.out(ASM_STA);
		C.code.out_addr_data(reg[1]->value);
		}
	if (reg[0]) {
		C.code.out(ASM_PLA);
		C.code.out(ASM_STA);
		C.code.out_addr_data(reg[0]->value);
		}
	}
}




// An increment or decrement statement.
//
// e.g. inc counter
//	++counter
//      dec counter
//	--counter
//	counter++
//	counter--
//
// This is a statement rather than an expression,
// so it isn't like a true 'C' inc/dec operator.
// To do that, we should do this in the expr()
// routine instead. [bj 09oct2006]
//
void upl_Compiler::inc_statement(
	Flex&			L,
	upl_Context& 		C,
	boolean			Prefix,
	const upl_Symbol       *S,
	upl_Subscript	       *Subscript)
{
#ifdef NO_WARNINGS
  byte	opcode = 0;
#else
  byte	opcode;
#endif
  upl_Symbol const *D;
  upl_Subscript	subscript;



  if (Prefix)
    Select(L.matches("inc") or L.matches("++"))	opcode = ASM_INC;
      when(L.matches("dec") or L.matches("--"))	opcode = ASM_DEC;
      otherwise
	abend(WHERE0, "Bad case");
    endsel


  if (S == NULL)
    {
    D = C.symbols.get(L, L.get_id(), true);

    if (D->count != 0)
      subscript.read(L, C, *D);
    }
  else
    {
    D = S;

    if (Subscript)
      {
      assert(D->count != 0);
      subscript = *Subscript;
      }
    else
      assert(D->count == 0);
    }


  if (not Prefix)
    Select(L.matches("++"))	opcode = ASM_INC;
      when(L.matches("--"))	opcode = ASM_DEC;
      otherwise
	abend(WHERE0, "Bad case");
    endsel


  /*
  if (D->in_local_storage)
    L.parse_error("In this implementation you cannot increment or "
		  "decrement local variables or parameters.");
  */


  if (D->symbol_type == upl_variable)
    {
    if (D->count == 0)
      reg_var(C, *D, NULL, false, false, 0, opcode);
    else
      {
      // Load into A, X.
      //
      reg_var(C, *D, &subscript, true);

      switch (opcode)
	{
	case ASM_INC:
	  C.code.out(ASM_CLC);
          C.code.out(ASM_ADC_IMM);
	  C.code.out(1);

	  if (D->value_bytes > 1)
	    {
	    C.code.out(ASM_BCC);
	    C.code.out(1);
	    C.code.out(ASM_INX);
	    }
	  break;

	case ASM_DEC:
	  C.code.out(ASM_SEC);
	  C.code.out(ASM_SBC_IMM);
	  C.code.out(1);

	  if (D->value_bytes > 1)
	    {
	    C.code.out(ASM_BCS);
	    C.code.out(1);
	    C.code.out(ASM_DEX);
	    }
	  break;

	default:
	  abend(WHERE0, "Bad case");
	}


      // Store A, X.
      //
      reg_var(C, *D, &subscript, false);
      }
    }
  else
    L.parse_error("Variable expected");
}




// Read an expression.
//
upl_value_type upl_Compiler::expr(
	Flex& 			L,	  // Read expression from here.
	upl_Context& 		C,
	upl_Expr_result&	Result,	  // Result of this expression; a type and perhaps constant value
	int			Depth,	  // Increasing order of precedence/convention/BODMAS
	boolean			Mandatory_id)
{
  upl_Expr_result	clause;
  boolean		swap_parms	= false;

  // These are the runtime subroutines we use to and, or, multiply, etc. our expressions.
  // <op_bb>  is for signed byte expressions.
  // <op_ww>  is for signed word expressions.
  // <op_ubb> is for unsigned byte expressions.
  // <op_uww> is for unsigned word expressions.
  //
#ifdef NO_WARNINGS
  upl_addr		op_bb = 0, op_ww = 0, op_ubb = 0, op_uww = 0;
#else
  upl_addr		op_bb, op_ww, op_ubb, op_uww;
#endif

  // The enumerated operation in this expression; e.g. multiply.
  //
  upl_op		op;

  // Is this enumerated operation signed (positive or negative) or unsigned (positive only)?
  //
  boolean		is_unsigned;


  // Remember the starting address in case we're able to optimise
  // the expression and can thus rollback the code.
  //
  upl_Context_state	expr_state;
  C.mark(expr_state);


  // If a binary operator...
  //
  if (Depth <= 4)
    {
    // Get the first part of the expression; e.g. >>A<< * b
    //
    expr(L, C, Result, Depth+1, Mandatory_id);


    // See if there is an operator; e.g. A >>*<< b
    //
    loop
      {
      op = upl_op_none;

      // We don't know the type of the expressions yet,
      // so we must remember subroutines for byte and word terms.
      //
      switch (Depth)
	{
	case 0:
	  //
	  // Logical and Bitwise or are the same...
	  //
	  if (L.matches("or") or L.matches("bitor"))
	    {
	    // The expression operand. 
	    // 
	    // We use this to precalcuate if possible at compile-time;
	    // We can only do this if the operands are constants.
	    //
	    op		= upl_or;

	    // These are the runtime subroutines we will call.
	    //
	    op_ubb 	= op_bb 	= RUNTIME_OR_BB;    // byte-size "or"
	    op_uww	= op_ww 	= RUNTIME_OR_WW;    // word-size "or"
	    }
	  break;

	case 1:
	  //
	  // Logical and Bitwise and are the same...
	  //
	  if (L.matches("and") or L.matches("bitand"))
	    {
	    // The expression operand. 
	    // 
	    // We use this to precalcuate if possible at compile-time;
	    // We can only do this if the operands are constants.
	    //
	    op		= upl_or;

	    // These are the runtime subroutines we will call.
	    //
	    op_ubb 	= op_bb 	= RUNTIME_AND_BB;    // byte-size "and"
	    op_uww	= op_ww 	= RUNTIME_AND_WW;    // word-size "and"
	    }
	  break;

	case 2:
	  // 
	  // Comparison.
	  //
	  Select(L.matches("=="))
	      op	= upl_eq;
	      op_ubb 	= op_bb 	= RUNTIME_EQ_BB;
	      op_uww	= op_ww 	= RUNTIME_EQ_WW;
	    when(L.matches("!="))
	      op	= upl_ne;
	      op_ubb 	= op_bb 	= RUNTIME_NE_BB;
	      op_uww	= op_ww 	= RUNTIME_NE_WW;
	    when(L.matches("<="))
	      op	= upl_le;
	      op_ubb 	= RUNTIME_ULE_BB;
	      op_bb 	= RUNTIME_LE_BB;
	      op_uww	= RUNTIME_ULE_WW;
	      op_ww 	= RUNTIME_LE_WW;
	    when(L.matches("<"))
	      op	= upl_lt;
	      op_ubb 	= RUNTIME_ULT_BB;	// Unsigned 
	      op_bb 	= RUNTIME_LT_BB;
	      op_uww	= RUNTIME_ULT_WW;
	      op_ww 	= RUNTIME_LT_WW;
	    when(L.matches(">"))
	      op	= upl_gt;
	      op_ubb 	= RUNTIME_UGT_BB;
	      op_bb 	= RUNTIME_GT_BB;
	      op_uww	= RUNTIME_UGT_WW;
	      op_ww 	= RUNTIME_GT_WW;
	    when(L.matches(">="))
	      op	= upl_ge;
	      op_ubb 	= RUNTIME_UGE_BB;
	      op_bb 	= RUNTIME_GE_BB;
	      op_uww	= RUNTIME_UGE_WW;
	      op_ww 	= RUNTIME_GE_WW;
	  endsel
	  break;

	case 3:
	  Select(L.matches("+"))
	      op		= upl_add;
	      op_ubb 	= op_bb 	= RUNTIME_ADD_BB;
	      op_uww	= op_ww 	= RUNTIME_ADD_WW;
	    when(L.matches("-"))
	      op		= upl_sub;
	      op_ubb 	= op_bb 	= RUNTIME_SUB_BB;
	      op_uww	= op_ww 	= RUNTIME_SUB_WW;
	  endsel
	  break;


	// Multiplication, etc.
	//
	case 4:
	  Select(L.matches("*"))	      // Multiply
	      op	= upl_mul;
	      op_ubb 	= -1;
	      op_bb 	= -1;
	      op_uww	= RUNTIME_UMUL_WW;
	      op_ww 	= RUNTIME_MUL_WW;
	    when(L.matches("/"))	      // Divide
	      op	= upl_div;
	      op_ubb 	= -1;
	      op_bb 	= -1;
	      op_uww	= RUNTIME_UDIV_WW;
	      op_ww 	= RUNTIME_DIV_WW;
	    when(L.matches("%"))	      // Modulo aka Remainder.
	      op	= upl_mod;
	      op_ubb 	= -1;
	      op_bb 	= -1;
	      op_uww	= RUNTIME_UMOD_WW;
	      op_ww 	= RUNTIME_MOD_WW;
	  endsel
	  break;
	}


      // If we had a binary operand...
      //
      if (op != upl_op_none)
	{
	// Get the next part of the expression; A * >>B<<
	//
	expr(L, C, clause, Depth+1, Mandatory_id);


	boolean	release_clause = false;
	boolean release_result = false;


	// If both clauses are constants, we can compute this at compile time.
	// We won't need to generate code to do this at run-time; This is more efficient.
	//
	if (Result.constant)
	  if (clause.constant)
	    {
	    switch (op)
	      {
	      case upl_xor: Result.value  = Result.value xor clause.value; break;
	      case upl_and: Result.value  = Result.value and clause.value; break;
	      case upl_or:  Result.value  = Result.value or  clause.value; break;
	      case upl_eq:  Result.value  = Result.value ==  clause.value; break;
	      case upl_ne:  Result.value  = Result.value !=  clause.value; break;
	      case upl_ge:  Result.value  = Result.value >=  clause.value; break;
	      case upl_gt:  Result.value  = Result.value >   clause.value; break;
	      case upl_le:  Result.value  = Result.value <   clause.value; break;
	      case upl_lt:  Result.value  = Result.value <=  clause.value; break;
	      case upl_mul: Result.value  = Result.value *   clause.value; break;

	      case upl_div:
		if (clause.value)
		  Result.value  = Result.value / clause.value;
		else
		  L.parse_error("Division by zero");
		break;

	      case upl_mod:
		if (clause.value)
		  Result.value  = Result.value % clause.value;
		else
		  L.parse_error("Modulo division by zero");
		break;

	      case upl_add: Result.value  = Result.value +   clause.value; break;
	      case upl_sub: Result.value  = Result.value -   clause.value; break;
	      default:
		abend(WHERE0, "Bad case");
	      }
	    }
	  else
#ifndef NEW_CODE
	    {
	    release_result = true;
	    C.code.push_value(Result.value, clause.value_bytes == 2);

	    switch (op)
	      {
	      case upl_ge:  op = upl_le; break;
	      case upl_gt:  op = upl_lt; break;
	      case upl_le:  op = upl_ge; break;
	      case upl_lt:  op = upl_gt; break;

	      case upl_div:
	      case upl_mod:
		swap_parms = true; break;

	      default:
		;//abend(WHERE0, "Bad case");	// But why? [20sep1998 bj]
	      }
#else
	    {
	    release_result = true;

	    // Push the constant value <Result.value> on the stack.
	    // If this clause we are about to be merged with using
	    // the <op> operation is a (2 byte) word, then we push
	    // it as a word.  [bj 14may2005]
	    //
	    C.code.push_value(Result.value, clause.value_bytes == 2);

	    // If we did push the <Result> value as a word,
	    // then convert <Result> itself into word value.
	    // If we don't do this, it'll be cast later on.
	    // [bj 14may2005]
	    //
	    if (Result.value_bytes == 1 && clause.value_bytes == 2)
	      {
	      // By changing <value_bytes> to 2, we are saying that
	      // this value has been stored using 2 bytes.
	      //
	      Result.value_bytes  = 2;

	      // If <Result> is a 8-bit char, assume a signed 16-bit integer.
	      // Otherwise assume an unsigned 16-bit integer. [bj 14may2005]
	      //
	      Result.value_type	  = Result.value_type == upl_char
				  ? upl_short
				  : upl_ushort;
	      }

	    switch (op)
	      {
	      case upl_ge:  op = upl_le; break;
	      case upl_gt:  op = upl_lt; break;
	      case upl_le:  op = upl_ge; break;
	      case upl_lt:  op = upl_gt; break;

	      case upl_div:
	      case upl_mod:
		swap_parms = true; break;

	      default:
		;//abend(WHERE0, "Bad case");	// But why? [20sep1998 bj]
	      }
#endif
	    // Exchange the two clauses so we know who is on top.
	    //
	    upl_Expr_result	temp;

	    swap(clause, Result, temp);
	    }
	else
	  //
	  // Result.variable * clause.constant ->
	  // Result.variable * clause.variable
	  //
	  if (clause.constant)
	    {
	    C.code.push_value(clause.value,
		clause.value_bytes < Result.value_bytes or
		clause.value_bytes == 2 or
		Result.value_bytes == 2);

	    if (clause.value_bytes < Result.value_bytes)
	      {
	      clause.value_bytes = Result.value_bytes;
	      clause.value_type  = Result.value_type;
	      }

	    release_clause = true;
	    }
	  else
	    //
	    // Result.variable * clause.variable
	    //
	    ;


	boolean generate = not Result.constant or not clause.constant;


	// Optimised
	if (generate and C.optimise_peephole)
	  if (Result.is_optimal() and clause.is_optimal())
	    generate = not optimise(C, op, Result, clause, expr_state);


	// Stack virtual machine.
	//
	if (generate)
	  {
	  if (release_clause)
	    clause.constant = false;

	  if (release_result)
	    Result.constant = false;


	  Result.ignore_transient();
	  Result.ignore_variable();


	  ushort	parm_bytes;

	  upl_value_type new_value =
	    normalise_binary(L, C, op,
		Result.value_type, clause.value_type, is_unsigned,
		&parm_bytes);


	  if (swap_parms)
	    {
	    C.code.out(ASM_JSR);
	    C.code.out_word_patch(
		parm_bytes == 1 ? RUNTIME_SWAP_BB : RUNTIME_SWAP_WW);

	    swap_parms = false;
	    }


	  long external_id =
		parm_bytes == 1?
			(is_unsigned ? op_ubb : op_bb) :
			(is_unsigned ? op_uww : op_ww);

	  if (external_id < 0)
	    abend(WHERE0, "Invalid opcode generated in expression");


	  C.code.out(ASM_JSR);
	  C.code.out_word_patch(external_id);


	  // Set the new value attributes.
	  //
	  if (new_value != Result.value_type)
	    {
	    Result.value_bytes	= value_bytes(new_value);
	    Result.value_type	= new_value;
	    }
	  }
	}
      else
	break;
      }
    }
  else
    {
    boolean	bracket = false;


    Select(L.matches("not") or L.matches("bitnot"))
	op 	= upl_not;
	op_bb 	= RUNTIME_NOT_BB;
	op_ww 	= RUNTIME_NOT_WW;

      when(L.peek_type() == flex_punc and L.matches("-"))
	op 	= upl_neg;
	op_bb 	= RUNTIME_NEG_BB;
	op_ww 	= RUNTIME_NEG_WW;


      when(L.peek_type() == flex_punc and equal(L.peek(), "("))
	op 	= upl_op_none;

	term(L, C, Result, Mandatory_id);

	bracket	= true;
	/*
	expr(L, C, Result, 0, Mandatory_id);
	L.check(")");
	bracket	= true;
	*/

      /*
      when(L.matches("("))
	op 	= upl_op_none;
	expr(L, C, Result, 0, Mandatory_id);
	L.check(")");
	bracket	= true;
        */

      otherwise
	op 	= upl_op_none;
    endsel


    if (not bracket)
      {
      term(L, C, Result, Mandatory_id);


      if (op != upl_op_none)
	if (Result.constant) {
	  switch (op)
	    {
	    case upl_not:
	      {
	      Result.value  = ~Result.value;

	      // If the result is a byte value,
	      // then clear the most significant byte.
	      // [bj 08jun2005]
	      //
	      if (Result.value_bytes == 1)
		Result.value &= 0xffu;
	      }
	    break;
	    case upl_neg:
	      {
	      Result.value  = -Result.value;

	      // If the result is a byte value,
	      // then clear the most significant byte.
	      // [bj 08jun2005]
	      //
	      if (Result.value_bytes == 1)
		Result.value &= 0xffu;
	      }
	    break;
	    default:
	      L.parse_error("Huh");
	      abend(WHERE0, "Bad case");
	    }
	}
	else
	  {
	  C.code.out(ASM_JSR);
	  C.code.out_word_patch(Result.value_bytes == 1 ? op_bb : op_ww);
	  }
      }
    }


  return Result.value_type;
}





// Terms are the atomic parts of an expression;
// they are the numbers or variables at the very bottom.
// The operators (e.g. mutiply, and) are higher up.
//
upl_value_type upl_Compiler::term(
	Flex& 			L,
	upl_Context& 		C,
	upl_Expr_result&	Result,
	boolean			Mandatory_id)
{
  // If a number, make it an integer constant.
  // 
  Select(L.peek_type() == flex_int or L.peek_type() == flex_hex)
      Result.set_constant(L, L.get_int());


    // If a character, also make it a character constant.
    //
    when(L.peek_type() == flex_char)
      char ch = *L.get(); // get first (and only) character of token.

      // Optionally convert to Commodore characacter set.
      //
      if (C.char_conversion != upl_char_conversion_none)
	ch = upl_Compiler::convert_char(ch, C.char_conversion);

      Result.set_constant(L, ch);
      Result.value_type  = upl_char;
      Result.value_bytes = 1;


    // The getch() function gets a character from the keyboard.
    //
    when(L.matches("getch"))
      // No arguments, but it has an empty ()
      L.check("(");
      L.check(")");

      // Call the runtime routine to get a character from the keyboard
      // and put it in <A>.
      //
      C.code.out(ASM_JSR);
      C.code.out_word_patch(RUNTIME_GET_CH_A);

      // Remember the state we are in now, where the character is in <A>.
      //
      upl_Context_state	key_state;
      C.mark(key_state);

      // Remember that we have stored the character in the accumulator 'a'.
      // We call this <*transient_a>. (Later we may decide to rollback
      // to this point...)
      //
      Result.set_value_type(upl_char, 0, upl_transient_a, &key_state);

      // For now though, we push it on the datastack.
      //
      C.code.out(ASM_JSR);
      C.code.out_word_patch(RUNTIME_PUSH_B);


    // If an amphersand, an address follows...
    //
    when(L.matches("&"))
      //
      // Get the symbol <D> we are taking the address of...
      //
      const upl_Symbol *D = C.symbols.get(L, L.get_id(), false);

      boolean		subscripted = false;  // A subscripted array? Assume not.
      upl_Subscript	subscript;

      // If a square bracket follows, this is an array reference.
      // 
      if (equal(L.peek(), "["))
	{
	subscripted = D->count != 0;  // If non-zero, then this is an array of that many elements.    
	if (subscripted)	      // So if this is an array....
	  {
	  subscript.read(L, C, *D);   // Read the value of the subscript and push it on the stack.
				      // This reads the enclosing square brackets too.
	  }
	}

      // Call push_var_addr() to push the address of the subscripted data on the stack.
      //
      push_var_addr(L, C, *D, subscripted ? &subscript : NULL, Result);


    // Memory Function.
    //
    // A UPL memory statement or C '*' pointer operator...
    //
    // CAUTION: 
    //       For pointers this works, but it isn't very flexible or easy to understand.
    //       There should be a much more elegant way of handling indirection.
    //	     This would be much easier to understand with a rewrite.
    //	     I noticed and fixed up some problems just before release; this section NEEDS TESTING.
    //	     [bj 09oct2006]
    //
    when(equal(L.peek(), "memory") or
	 equal(L.peek(), "mem") or
	 equal(L.peek(), "*"))
      upl_value_type 	vtype	= upl_byte; // The type of the location we are looking up.
      int	       	vbytes	= 1;	    // How many bytes to read from that location (should match type).

      // Remember what type of memory function.
      //
      boolean pointer_assignment = L.matches("*");    // 'C' pointer 
      if (not pointer_assignment)
	if (not L.matches("mem"))
	  L.check("memory");

      upl_Context_state	key_state;
      upl_Expr_result	clause;

      if (not pointer_assignment)
	{
	// If a UPL "memory" function
	//

	// Get the type (e.g. "memory.short") so we know 
	// how many bytes we will be accessing.
	//
	if (L.matches("."))
	  {
	  const upl_Symbol *T = type(L, C, true);
	  vtype  = T->value_type;   // We will access the memory as this type
	  vbytes = T->value_bytes;  // And this many bytes.
	  }

	L.check("[");	// "memory[" now needs that opening square bracket

	expr(L, C, clause, 0);	  // get the memory location.
	}
      else
	{
	// 
	// Otherwise this is a C-style pointer assignment.
	// 
	// Get the memory address we will be accessing.
	//
	term(L, C, clause, Mandatory_id);

	// FIX: If the pointer is a symbol, then we take its type.
	//	Otherwise we stick to byte. [bj 09oct2006]
	//
	if (clause.is_variable)
	  {
	  vtype  = clause.value_type;
	  vbytes = clause.value_bytes;

	  // Since this is a variable, we don't know what address we will
	  // be looking up. Accordingly we can't use the "if (clause.constant)"
	  // block below, since that assumes we know the address.
	  }
	else
	  //
	  // We could allow this e.g. *(200) to look up memory location 200,
	  // but it's not a very clean way of doing it.  Best for now to
	  // refuse to do this until we get a better pointer system in place.
	  //
	  L.parse_error("Cannot use non-pointer for pointer access");
	}


      // If this wasn't a C-style pointer, get the closing bracket.
      //
      if (not pointer_assignment)
	L.check("]");


      // If the memory address is a constant, we know the address at compile time.
      // We can optimise access now. This is good.
      //
      // FIX:	This assumes we know the actual location we will be looking up.
      //	That isn't the case for pointer variables, so we exclude those here.
      //        [bj 09oct2006]
      //
      // TODO:	The array/pointer code here and above needs a rewrite. 
      //	Its inelegant and confusing. [bj 09oct2006]
      //
      if (clause.constant && !pointer_assignment)
	{
	// At some point here, the value will be in the <A> register.
	// If a word, it will also be in the <X> register.
	// But for now assume only <A>. [bj 09oct2006]
	//
	upl_transient_type transient_type = upl_transient_a;

#ifndef NEW_CODE_MEMACCESS
	//
	// Load the accumulator <A> from this address.
	//
	// NOTE: This isn't loading the X register which would be wrong
	//	 when accessing short pointers, etc. So we'll use the
	//	 NEW_CODE_MEMACCESS code instead. [bj 09oct2006]
	//
	C.code.out(ASM_LDA);
	C.code.out_word(clause.value);
#else
	// Harry's New Memory Access Code.
	//
	// This handles 8 and 176 bit operands.
	// Comments are mine. [bj 09oct2006]

	//
	// If it in the first 256 bytes of memory, we can use zero-page addressing.
	//
	if (0 <= clause.value && clause.value <= 255)
	  {
	  // Load the 'A' register from this zero page location.
           C.code.out(ASM_LDA_Z);
           C.code.out((byte)(clause.value & 0xff));

	   // If a word, load the second byte into the X register.
	   //
	    if (vbytes == 2) 
	      {
	      C.code.out(ASM_LDX_Z);
	      C.code.out((byte)((clause.value & 0xff) +1));

	      transient_type = upl_transient_ax;	// FIX: Must remember transient is now A and X [bj 09oct2006]
	      }
	  }
	else
	  {
	  // Otherwise if not in zero page, do the same from a full absolute address.
	  // 
           C.code.out(ASM_LDA);
           C.code.out_word(clause.value);
	   
	   // If a word, load the X into this second register.
	   //
	    if (vbytes == 2) 
	      {
	      C.code.out(ASM_LDX);
	      C.code.out_word(clause.value + 1);

	      transient_type = upl_transient_ax;	// FIX: Must remember transient is now A and X [bj 09oct2006]
	      }
	  }
#endif
	
	// Remember Transient State.
	//
	// Remember our situation in case we want to roll back later.
	//
	C.mark(key_state);

	// WAS [bj 09oct2006]: Result.set_value_type(upl_byte, 0, upl_transient_a, &key_state);
	//  EXPLAIN: Since we may have stored in <X> too, we need to reflect that, etc.[bj 09oct2006]

	// The value we have loaded into <A> or <A, X> is whatever we specified,
	// as stored in <transient_type>. We may want to rollback to this point later.
	//
	Result.set_value_type(vtype, 0, transient_type, &key_state);
	C.code.out(ASM_JSR);


#ifdef NEW_CODE_MEMACCESS
	//
	// The value is now in the register <A> and perhaps <X>.
	// Push these on the stack.
	//
	C.code.out_word_patch(
		vbytes == 1 ? RUNTIME_PUSH_B : RUNTIME_PUSH_W);
	Result.value_bytes = vbytes;
#else
	C.code.out_word_patch(RUNTIME_PUSH_B);
#endif
	}
      else
	{
	// Indirect Memory address.
	//
	// Location is unknown at compile time.
	// We must generate code to do this at runtime.
	//
	ensure_pushed(L, C, clause);	// Make sure <clause> is pushed on stack.
	C.code.out(ASM_JSR);

	// If a C pointer, get the value type 
	//
	if (pointer_assignment)
	  {
	  // If a pointer variable, convert from e.g. pointer_short to just short.
	  //
	  vtype  = pointer_value_type(clause.value_type);
	  vbytes = value_bytes(vtype);
	  }

	// Generate code to peek at that memory location,
	// and push the result on the stack.
	// 
	C.code.out_word_patch(
		vbytes == 1 ? RUNTIME_PEEK_W_B : RUNTIME_PEEK_W_W);

	// Because this must be done at runtime, we don't know the actual result,
	// so expr() won't be able to precalculate the results.
	// However we do return the type for type-checking.
	//
	Result.set_value_type(vtype, 0);
	}


    // Return true iff the given symbol is defined...
    // e.g. if defined(foobar) 
    //
    when(L.matches("defined"))
      L.check("(");
      const upl_Symbol *D = C.symbols.get(L, L.get_id(), false);
      L.check(")");

      Result.set_constant(L, D != NULL);
      Result.value_type  = upl_boolean;
      Result.value_bytes = 1;


    // Push a Variable (possibly subscripted) or call a function.
    //
    // TODO: The array/pointer code here and above needs a rewrite. 
    //	     Its inelegant and confusing. [bj 09oct2006]
    //
    when(L.peek_type() == flex_id)
      //
      // Find out what this symbol <D> is...
      //
      const upl_Symbol *D = C.symbols.get(L, L.peek(), Mandatory_id);


      if (D != NULL)
	switch (D->symbol_type)
	  {
	  //
	  // A regular variable.
	  //
	  case upl_variable:
	    {
	    boolean		subscripted = false;  // Assume no subscript
	    upl_Subscript	subscript;
	    boolean		is_array = false;     // Assume not an array

	    L.get_id(NULL);

	    // If an array or pointer, set <is_array> truee.
	    //
	    is_array = D->count != 0 or is_pointer(D->value_type);
	    if (is_array)
	      if (equal(L.peek(), "["))	    // If a square bracket, read in subscript.
		{
		subscripted = true;
		subscript.read(L, C, *D);
		}

	      /* COUT UNSURE Has this been fixed? [bj 09oct2006]
	      // If it's going to say it's a variable, then it shouldn't push any code.  [bj 15sep98]
	      // [bj 20sep98] now it's never getting pushed... grrr!
		*/


	    // If not an array or it is a non-subscripted pointer,
	    // then load the variable into <A/X> as appropriate.
	    // (If it is a pointer, we're using its value just
	    //  like any other variable; e.g. we may be about
	    //  to add to it or something).
	    //
	    if (not is_array or (is_pointer(D->value_type) && !subscripted))
	      {
	      Result.set_variable(*D);
	      ensure_pushed(L, C, Result);	// UNSURE: [bj 20sep98]
	      }
	    //
	    // If we have a subscript... we always do arrays like this,
	    // but if the pointer had a subscript then we treat it 
	    // like an array too.
	    //
	    else if (subscripted)
	      push_var(C, *D, &subscript, &Result);
	    else
	      //
	      // Otherwise it's a regular push-variable.
	      //
	      push_var_addr(L, C, *D, NULL, Result);
	    }
	    break;


	  // Constant
	  //
	  case upl_constant:
	    Result.set_constant(L, D->value);
	    Result.value_type  = D->value_type;
	    Result.value_bytes = D->value_bytes;
	    L.get_id(NULL);
	    break;


	  // Call a subroutine.
	  //
	  case upl_function:
	    L.get_id(NULL);
	    subroutine_statement(L, C, *D, &Result);
	    break;



	  // Procedures (voids) cannot be called from expressions.
	  //
	  case upl_procedure:
	    L.parse_error("A procedure cannot be called from "
			  "within an expression");
	    break;


	  // They can access labels just like any other constant.
	  // Of course, should be careful how they use them!
	  //
	  case upl_label:
	    Result.set_constant(L, D->value);
	    L.get_id(NULL);
	    break;


	  default:
	    L.parse_error("Expression term expected [1]");
	  }
	else
	  {
	  // UNSURE:  Is this a good idea? Presumably this is for the assembler,
	  //	      but for the compiler does that mean we'd let them reference
	  //	      variables that haven't been defined?  [bj 09oct2006]
	  // SUGGEST: Have an error if called from the compiler,
	  //	      but allow from the assembler. [bj 09oct2006]
	  //

	  //
	  // Assume zero.
	  //
	  Result.set_constant(L, 0);
	  L.get_id(NULL);
	  }


      // If brackets, then we either have a cast of a subexpression inside brackets.
      //
    when(L.matches("("))
      {
      // See if the next token is a type.
      //
      const upl_Symbol *T = type(L, C, false);

      // If it is, we have a cast...
      // 
      if (T != NULL)
	{
	L.check(")");

	// Get the value we are going to cast...; e.g. (byte) >>45<<
	//
	upl_value_type subexpr_type = term(L, C, Result, Mandatory_id);

	// Make sure that value is pushed on the stack.
	//
	ensure_pushed(L, C, Result);

	// Now call the code to cast that pushed value into the desired type <T>.
	//
	normalise_unary(L, C, T->value_type, subexpr_type);
	Result.value_type  = T->value_type;
	Result.value_bytes = T->value_bytes;
	}
      else
	{
	// Otherwise we have a subexpression.
	//
	expr(L, C, Result, 0);

	L.check(")");
	}
      }


    otherwise
      L.parse_error("Expression term expected [2]");
  endsel


  return Result.value_type;
}



void upl_Compiler::push_var_addr(
	Flex&		 L,
	upl_Context& 	 C,
  const	upl_Symbol&  	 D,
	upl_Subscript   *Subscript,
	upl_Expr_result& Result)
{
  upl_Context_state	state;


  // If a variable is in local storage, then it is a so-called automatic variable.
  // It lives on the data stack.  (See UPLRTIME.ASM to learn how stack frames work).
  //
  Select(D.in_local_storage)
      C.code.out(ASM_LDA_Z);
      C.code.out(RUNTIME_BP);

      C.code.out(ASM_SEC);		// +1
      C.code.out(ASM_ADC_IMM);
      C.code.out(D.is_local);

      C.code.out(ASM_LDX_IMM);
      C.code.out_word_patch(
	RUNTIME_DATASTACK, upl_patch_method_set_high);

      C.mark(state);

      if (is_pointer(D.value_type))
	Result.set_value_type(
		D.value_type, 0,
		upl_transient_ax, &state);
      else
        Result.set_value_type(
		convert_pointer(L, D.value_type), 0,
		upl_transient_ax, &state);

      C.code.out(ASM_JSR);
      C.code.out_word_patch(RUNTIME_PUSH_W);


    // Push a variable using a subscript.
    //
    when(Subscript)
      //
      // If the subscript is constant, then we can calculate the address at compile time.
      //
      Select(Subscript->use_constant)
	  Result.set_constant_pointer(L, D.value_type, D.value+Subscript->index.value*D.value_bytes);

	// If subscript is dynamic (not known at compile time), we have to generate the
	// code to push the offset on the stack, calculate its location and then load that.
	//
	otherwise
	  // Subscript expression is sitting on the stack.
	  //

	  // If size is other than 1 byte per item, we nust scale.
	  // (NOTE: could make this more compact! [bj 09oct2006]) 
	  //
	  if (D.value_bytes != 1)   
	    {
	    C.code.out(ASM_LDA_IMM);	      // <A/X> = size of eeach item.
	    C.code.out(D.value_bytes);
	    C.code.out(ASM_LDX_IMM);
	    C.code.out(0);
	    C.code.out(ASM_JSR);
	    C.code.out_word_patch(RUNTIME_PUSH_W);  // Push that on stack 
	    C.code.out(ASM_JSR);
	    C.code.out_word_patch(RUNTIME_UMUL_WW); // Multiply with subscript index already on stack
	    }

	  // Push the array/pointer start on the stack.
	  //
	  // UNSURE: Not sure pointers will work this way,
	  //	     since <D.value> for arrays is the start of the array.
	  //	     For pointers, it is the address of the pointer.
	  //	     Do both arrays and pointers work here?  TEST. [bj 09oct2006]
	  //
	  C.code.out(ASM_LDA_IMM);
	  C.code.out_word(D.value, upl_data_byte_lo);
	  C.code.out(ASM_LDX_IMM);
	  C.code.out_word(D.value, upl_data_byte_hi);
	  C.code.out(ASM_JSR);
	  C.code.out_word_patch(RUNTIME_PUSH_W);

	  // Add the offset with the base.
	  //
	  C.code.out(ASM_JSR);
	  C.code.out_word_patch(RUNTIME_ADD_WW);

	  // On the stack, we now have the value the pointer was pointing too.
	  // As we can't predict what this will be, there is no transient.
	  //
	  Result.set_value_type(convert_pointer(L, D.value_type), 0);
	endsel


    otherwise
      //
      // Otherwise we want to push the value stored at the variable location.
      // Effectively, <D.value> which is address we look up like a literal pointer,
      // and push the result on the stack.
      //
      Result.set_constant_pointer(L, D.value_type, D.value);
  endsel
}





void upl_Compiler::push_var(
	upl_Context& 	 C,
  const	upl_Symbol&  	 D,
	upl_Subscript   *Subscript,
	upl_Expr_result	*Result)
{
  upl_Context_state	state;


  // Load the variable <D> with optional subscript <Subscript>.
  // This will load the contents into registers <A/X> (ie. <A> or <A and X>).
  //
  reg_var(C, D, Subscript, true);


  // Remember the state if we want to rollback to now,
  // when we hold the registers.
  //
  if (Result)
    C.mark(state);


  switch (D.value_bytes)
    {
    case 2:
      //
      // Remember the value is currently in <A/X>
      //
      if (Result)
	Result->set_value_type(D.value_type, D.indirection,
		upl_transient_ax, &state);

      // Save it on the stack.
      //
      C.code.out(ASM_JSR);
      C.code.out_word_patch(RUNTIME_PUSH_W);
      break;

    case 1:
      //
      // Remember the value is currently in <A>
      //
      if (Result)
	Result->set_value_type(D.value_type, D.indirection,
		upl_transient_a, &state);

      // Save it on the stack.
      //
      C.code.out(ASM_JSR);
      C.code.out_word_patch(RUNTIME_PUSH_B);
      break;

    default:
      abend(WHERE0, "Bad case");
    }
}



void upl_Compiler::pop_var(
	upl_Context& 	C,
  const	upl_Symbol&  	D,
	upl_Subscript  *Subscript)
{
  // Pop the value off the stack and load it into <A/X>.
  //
  switch (D.value_bytes)
    {
    case 2:
      C.code.out(ASM_JSR);
      C.code.out_word_patch(RUNTIME_POP_W);
      break;

    case 1:
      C.code.out(ASM_JSR);
      C.code.out_word_patch(RUNTIME_POP_B);
      break;

    default:
      abend(WHERE0, "Bad case");
    }


  // Now store the registers in this variable.
  //
  reg_var(C, D, Subscript, false);
}




// Load/store variable into <A/X>.
//
// This is a generalised subroutine for accessing a variable.
//
// It load or stores (or otherwise accesses) the variable <D> or <D[Subscript]> 
// into registers <A/X>.  We can also do other operations through this, 
// such as increment, decrement, etc.
//
// IN:
//    C		    Context 
//    D		    The variable <D> we are accessing.
//    Subscript	    Optional.  If the value we are accessing is an element indexed
//		    from a pointer/array, the subscript used to access it is 
//		    passed here.
//    Load	    True  iff we are loading the value.
//	            False iff we are storing it.
//	            (We use <Other_opcode> if neither of these.)
//    Other_opcode  If neither loading or storing, we do this instead.
//		    Only certain opcodes are allowed here.
//    Have_value    True iff a <Value> has been passed.
//		    If not, we use the value in <A> or <A/X>.
//    Value	    The <Value> if specified.  (Ignored if we don't <Have_value>).
//    Address_value True iff <Value> is actually an address.
//    
//
void upl_Compiler::reg_var(
		upl_Context&	C,
	const 	upl_Symbol& 	D,
		upl_Subscript  *Subscript,
		boolean		Load,
		boolean		Have_value,
		long		Value,
		byte		Other_opcode,
		boolean		Address_value)
  {
  // Offset to element in array.
  //
  // This is only used if we are accessing an array, and we knew the offset
  // at compile time.  Otherwise we simply leave this as zero.
  //
  ulong    offset	= 0;		// offset in bytes

  // The opcode to access the value.
  //
  // Are we loading or storing this variable into the registers <A/X>?
  // By convention <A> is the low byte of the value, and <X> the high byte.
  // If the value is one byte only, then only <A> is used.
  //
  // NOTE: Although we call these <opcode_st*>, it isn't necessarily a store.
  //	   It might actually be a load, depending on <Load>.
  //	   It might be <Other_opcode> too. [bj 21jun2005]
  //
  byte	   opcode_sta	= Load ? ASM_LDA : ASM_STA,
	   opcode_stx	= Load ? ASM_LDX : ASM_STX,
	   opcode_test_a  = ASM_CMP;	// Opcode to test low byte.

  // What is the address we are to use to access the value?
  //
  ulong    access_addr	= D.value;

  // Are we using a zeropage pseudoregister to access the value?
  //
  boolean  zeropage	= false;

  // Do we need to test the value after we retrieve/change it?
  //
  boolean  post_test	= false;

  // The result of this expression <D [Script]>.
  //
  upl_value_type  etype;	// The datatype of <D>.
  int		  ebytes;	// The number of bytes a value of this type takes.


  // Is <D> a pointer (as opposed to an regular variable or fixed array).
  //
  boolean epointer = is_pointer(D.value_type);


  // If <D> is a pointer and we're accessing it through a <Subscript>,
  //
  if (Subscript != NULL && epointer)
    {
    // The result won't be the pointer, but what it is pointing too.
    // Convert <etype, ebytes> into what it is pointing too.
    //
    etype  = pointer_value_type(D.value_type);
    ebytes = value_bytes(etype);

    // The result is a regular data type 
    // (not an address or pointer value)
    //
    Address_value = false;
    }
  else
    {
    // Otherwise the expression will have the same attributes 
    // as the datatype of <D>.
    //
    etype  = D.value_type;
    ebytes = D.value_bytes;
    }


  // If another opcode. use that instead.
  //
  if (Other_opcode)
    {
    opcode_sta	=
    opcode_stx	= Other_opcode;

    // If we're decrementing we will want to test it after the operation
    // so the appropraiate register flags <C, Z, etc> are set.
    //
    if (Other_opcode == ASM_DEC)
      post_test = true;
    }


  // Subscript looking up a regular array or pointer.
  //
  // If we are using a <Subscript> to access the value...
  //
  if (Subscript != NULL)
    {
    // Note there are two ways we can access a value in an array.
    //
    // If the variable was declared as an array, then we know the address of 
    // the memory block already.  Looking up values in this sort of array is easy.
    //
    // If the variable is a pointer, then the pointer variable will contain
    // the address of the memory block at runtime.  Thus we must generate 
    // code to look it up at runtime.
    //


    // A Fast Array Lookup.
    //
    // Under certain conditions (the array being in a page, the index variable
    // being between 0 and 255, etc.) we can access the expression using
    // the pointer offset.
    //
    // Note we are using a regular array here, and *not* a pointer!
    //
    Select(Subscript->use_fast_variable && !epointer)
	{
	// ASSERT: We are using a variable for this fast lookup.
	//
	assert(Subscript->index.variable != NULL);

	// We must have page alignment for this to work,
	// so declare that in the context <C>.  [bj 21sep1998]
	//
	C.needed_alignment = true;

	// If it's a single byte, we use that as the offset.
	// This is trivial.
	//
	if (ebytes == 1)
	  {
	  C.code.out(ASM_LDY);
	  C.code.out_addr_data(Subscript->index.variable->value);
	  }
	else
	  {
	  // For a double-byte array, we'll need to multiply the offset by two.
	  // This is a bit more work...
	  // 

	  // If we don't <Have_value> in <Value>, then we have it in the <A> 
	  // or <A/X> registers.  We'll need to work with the <A> register here,
	  // so save it on the stack.
	  //
	  if (!Have_value)
	    C.code.out(ASM_PHA);

	  // Load the accumulator <A> with the index value.
	  // The index value is held in the memory address 
	  // <Subscript->index.variable->value>.
	  //
	  C.code.out(ASM_LDA);
	  C.code.out_addr_data(Subscript->index.variable->value);

	  // Now we double it by shifting it left...
	  //
	  C.code.out(ASM_ASL);

	  // Now we store that in the <Y> register ready to lookup.
	  //
	  C.code.out(ASM_TAY);

	  // Restore <A> off the stack.
	  // 
	  if (!Have_value)
	    C.code.out(ASM_PLA);    // UNSURE WAS [bj 21jun2005]: ASM_PHA 
	  }

	// Can't STX_Y, so must use TXA and STA_Y.
	//
	opcode_sta	= Load ? ASM_LDA_Y : ASM_STA_Y;
	opcode_stx	= Load ? ASM_LDX_Y : ASM_STA_Y;
	opcode_test_a	= ASM_CMP_Y;
	}


      // A constant subscript lookup.
      //
      // If <Subscript->use_constant> is true then the subscript is a constant.
      // If so and we're looking up a regular array (ie. not a pointer) with this
      // contstant value <Subscript->index.value>  then we can calculate the 
      // offset now, at compile time!
      //
      when(Subscript->use_constant && !epointer)
	{
	offset = Subscript->index.value * ebytes;
	}


      // Any other type of array or a pointer lookup.
      //
      otherwise
	{
	// If we don't <Have_value> in <Value>, then we have it in the <A> 
	// or <A/X> registers.  Save these registers on the stack.
	//
	// HFD to test for loading here to skip when  var = array[var]
	//
	if (!Have_value and !Load)
	  {
	  C.code.out(ASM_PHA);

	  if (ebytes == 2)
	    {
	    C.code.out(ASM_TXA);
	    C.code.out(ASM_PHA);
	    }
	  }

	// If this is a pointer lookup with a constant,
	// we will use the zeropage <CL/CH> to do an indirect memory
	// access to get the value...
	//
	if (epointer && Subscript->use_constant)
	  {
	  // HFD - slightly optimized
	  // no need to double a constant at runtime
	  if (ebytes == 2) {
		  C.code.out(ASM_LDA_IMM);
		  C.code.out(Subscript->index.value << 1);
		  C.code.out(ASM_STA_Z);
		  C.code.out(RUNTIME_CL);
		  C.code.out(ASM_LDA_IMM);
		  C.code.out(Subscript->index.value >> 7);
		  C.code.out(ASM_STA_Z);
		  C.code.out(RUNTIME_CH);
	  } else {

	  // The constant subscript index value is stored in <Subscript->index.value>.
	  //
	  // Store the LSB of constant subscript index in <CL>.
	  //
	  C.code.out(ASM_LDA_IMM);
	  C.code.out(Subscript->index.value);
	  C.code.out(ASM_STA_Z);
	  C.code.out(RUNTIME_CL);

	  // Store the MSB of constant subscript index in <CL>.
	  //
	  C.code.out(ASM_LDA_IMM);
	  C.code.out(Subscript->index.value>>8);
	  C.code.out(ASM_STA_Z);
	  C.code.out(RUNTIME_CH);
	  //
	  // The above code, when executed by the 6502, 
	  // will store the subscript index value in <CL, CH>.
	    }
	  }
	else
	  {
	  // Otherwise the subscript index expression is pushed on the stack.
	  // Pop that word straight into <CL/CH>.
	  //
	  C.code.out(ASM_JSR);
	  C.code.out_word_patch(RUNTIME_POP_W_C);
	  }

	// Assertion: The index value is now in <CL/CH>.
	//

	// Convert index value in <CL/CH> into an index byte offset.
	//
	// If the array/pointer has byte elements, then each element
	// is one byte long as so <CL/CH> already is a byte offset.
	//
	// But if it has word elements, then we need to multiply the 
	// index value by two to convert it into a byte offset.
	//
	//HFD  zpage are not words
	if (ebytes == 2 and not (epointer && Subscript->use_constant))
	  {
	  C.code.out(ASM_ASL_Z);
	  C.code.out(RUNTIME_CL);
	  C.code.out(ASM_ROL_Z);
	  C.code.out(RUNTIME_CH);
	  }


	// NOTE: If you wanted to add support for multidimensional arrays,
	//	 you would do it here.  Basically you would keep adding
	//	 and multiplying the offset to represent the indexes
	//	 to the other dimensions of the array until you end up
	//	 with the equivalent of an index to a 1-dimensional array.
	//	 [bj 21jun2005]


	// Calculate the address of the element we are accessing.
	//
	// <CL/CH> now holds the byte offset into the array/pointer.
	// To this we need to add the address of the array/pointer itself.
	// Add the two, and we'll have the address of the element
	// that we want to access.
	//

	// First the low byte <CL>...
	// 

	// If <D> is a pointer, then we don't know the address of the memory block
	// that the array is pointing to at compile time.  So, we have to get the
	// address of that memory block by looking up the pointer's value.
	// This is the address of the memory block.  The low byte of the pointer
	// value is the low byte we will add to <CL/CH>.
	//
	if (epointer)
	  {
	  C.code.out(ASM_LDA);
	  C.code.out_addr_data(D.value);
	  }
	else
	  {
	  // Otherwise it must be a regular array.
	  //
	  // For regular arrays, we do know the address of the memory block at
	  // compile time.  This is the low byte we will add to <CL/CH>.
	  //
	  C.code.out(ASM_LDA_IMM);
	  C.code.out_word(D.value, upl_data_byte_lo);
	  }

	// Assertion: The low byte of the memory block address is in <A>.

	// Now add <A> to <CL/CH>, clearing carry.
	//
	C.code.out(ASM_CLC);	// We are about to add, so clear carry.
	C.code.out(ASM_ADC_Z);
	C.code.out(RUNTIME_CL);
	C.code.out(ASM_STA_Z);
	C.code.out(RUNTIME_CL);

	// Assertion: The <carry> flag remembers if there is anything to carry here.
	//	      We will use this when we add the high byte.


	// Now the high byte <CH>...
	//

	// If a pointer, get the high byte of the address of the memory block
	// by looking up the high byte of the pointer value.
	//
	if (epointer)
	  {
	  C.code.out(ASM_LDA);
	  C.code.out_addr_data(D.value+1);
	  }
	else
	  {
	  // Likewise if a regular array, we already know the address of the 
	  // memory block, so we can load that straight into <A>.
	  //
	  C.code.out(ASM_LDA_IMM);
	  C.code.out_word(D.value, upl_data_byte_hi);
	  }


	// Now add <A> to <CH>, using the carry flag already set.
	//
	C.code.out(ASM_ADC_Z);
	C.code.out(RUNTIME_CH);
	C.code.out(ASM_STA_Z);
	C.code.out(RUNTIME_CH);


	// ------------------------------------------------------------
	// HFD  This is untrue - but we can't be sure that the index is
	//	below 128 so play it safe anyway
	// ------------------------------------------------------------

	// We are going to do an indirect zeropage access.
	// This uses the <Y> register
	//
	C.code.out(ASM_LDY_IMM);
	C.code.out(0);


	// If they are using another opcode besides load/store...
	//
	// You can't increment or decrement in this way,
	// so we must refuse.  (ELABORATE [bj 21jun2005])
	//
	switch (Other_opcode)
	  {
	  case ASM_INC:
	  case ASM_DEC:
	    abend(WHERE0, "Cannot INC/DEC subscripted variable");
	    break;

	  // Change the opcode to reflect the fact we'll now
	  // be using a zero-page Y-indexed instruction
	  // to access the element.
	  //
	  default:
	    opcode_sta	= Load ? ASM_LDA_IY : ASM_STA_IY;
	    opcode_stx	= Load ? ASM_LDA_IY : ASM_STA_IY;
	  }

	// If they're testing the array element value,
	// then they will be using this instead.
	//
	opcode_test_a	= ASM_CMP_IY;

	// We will do this through a zero-page operation on 
	// pseudo-registers <CL/CH>.
	//
	access_addr	= RUNTIME_CL;
	zeropage	= true;


	// Ok!  We're all ready to use the subscript.
	// Restore the <A> or <A/X> values we saved on the stack
	// so we can do this.
	//
	// HFD needs to test for loading here to skip when  var = array[var]
	//
	if (!Have_value and !Load)
	  {
	  C.code.out(ASM_PLA);

	  if (ebytes == 2)
	    {
	    C.code.out(ASM_TAX);
	    C.code.out(ASM_PLA);
	    }
	  }
	}
    endsel;
    }
  //
  // If we are not using a subscript, then this is an atomic variable lookup.
  //
  // Is this atomic variable on the datastack?
  //
  else if (D.in_local_storage)
    {
    // Do an indexed load/store to [datastacks+bp]
    //
    switch (Other_opcode)
      {
      case ASM_INC:
	{
	C.code.out(ASM_LDX_Z);
	C.code.out(RUNTIME_BP);
	opcode_sta	=
	opcode_stx	= ASM_INC_X;
	opcode_test_a	= ASM_CMP_X;
	}
	break;

      case ASM_DEC:
	{
	C.code.out(ASM_LDX_Z);
	C.code.out(RUNTIME_BP);
	opcode_sta	=
	opcode_stx	= ASM_DEC_X;
	opcode_test_a	= ASM_CMP_X;
	}
	break;

      default:
	{
	C.code.out(ASM_LDY_Z);
	C.code.out(RUNTIME_BP);
	opcode_sta	= Load ? ASM_LDA_Y : ASM_STA_Y;
	opcode_stx	= Load ? ASM_LDX_Y : ASM_STA_Y;
	opcode_test_a	= ASM_CMP_Y;
	}
      }

    }


  // Now, Generate the code to access the variable value.
  //
  switch (ebytes)
    {
    // A single-byte variable value?
    //
    case 1:
      {
      // Have they given us a value to load as the variable's value?
      //
      if (Have_value)
	{
	C.code.out(ASM_LDA_IMM);

	// Is the value an address?  If so, the linker will need to
	// relocate it, so we must flag it as <upl_data_byte_lo>.
	//
	if (Address_value)
	  C.code.out_word(Value, upl_data_byte_lo);
	else
	  //
	  // Otherwise simply write the low byte of the value.
	  //
	  C.code.out(Value);
	}

      // Now execute the instruction to access the <value>.
      //
      C.code.out(opcode_sta);	  // Remember this can be STA, LDA, INC, etc!
      if (D.in_local_storage)
	{
	// Because (data) stack points to top, we must increment.  
	// (ELABORATE [bj 21jun2005])
	//
	C.code.out_word_patch(
		RUNTIME_DATASTACK, upl_patch_method_add, D.value+1);
	}
      //
      // If a zeropage access, use the <access_addr> provided.
      //
      else if (zeropage)
	{
	C.code.out(access_addr);
	}
      else
	//
	// Otherwise add any <offset> to the <access_addr>.
	//
	// Note that if we couldn't calculate the <offset> at compile time,
	// this will simply be zero.
	//
	C.code.out_addr_data(access_addr+offset);
      }
      break;


    // A double-byte variable value?
    //
    case 2:
      {
      // Have they given us a value to load as the variable's value?
      //
      if (Have_value)
	{
	C.code.out(ASM_LDA_IMM);

	// Is the value an address?  If so, the linker will need to
	// relocate it, so we must flag it as <upl_data_byte_lo>.
	//
	if (Address_value)
	  C.code.out_word(Value, upl_data_byte_lo);
	else
	  //
	  // Otherwise simply write the low byte of the value.
	  //
	  C.code.out(Value);
	}

      // Now execute the instruction to access the <value>.
      //
      C.code.out(opcode_sta);	  // Remember this can be STA, LDA, INC, etc!
      if (D.in_local_storage)
	{
	// Because (data) stack points to top, we must increment.  
	// (ELABORATE [bj 21jun2005])
	//
	C.code.out_word_patch(
		RUNTIME_DATASTACK, upl_patch_method_add, D.value+1);
	}
      //
      // If a zeropage access, use the <access_addr> provided.
      //
      else if (zeropage)
	C.code.out(access_addr);
      else
	//
	// Otherwise add any <offset> to the <access_addr>.
	//
	// Note that if we couldn't calculate the <offset> at compile time,
	// this will simply be zero.
	//
	C.code.out_addr_data(access_addr+offset);


      // If <post_test> is on then they were doing an <Other_opcode> like DEC
      // and need to be able to test the result.
      //
      if (post_test)
	{
	C.code.out(ASM_LDA_IMM);
	C.code.out(0xffu);

	C.code.out(opcode_test_a);	// Test decremented value, etc.

	if (D.in_local_storage)
	  C.code.out_word_patch(
		  RUNTIME_DATASTACK, upl_patch_method_add, D.value+1);
	else if (zeropage)
	  C.code.out(access_addr);
	else
	  C.code.out_addr_data(access_addr+offset);
	}

	if (opcode_stx == ASM_STA_IY) {
		C.code.out(ASM_INY);			// HFD fix for array[var] = expr
		if (not Have_value)
			C.code.out(ASM_TXA);		// HFD fix for array[var] = var
	}
	if (opcode_stx == ASM_LDA_IY) {
		C.code.out(ASM_INY);			// HFD #1 fix for var = array[var]
		C.code.out(ASM_PHA);
	}



      if (Have_value)
	if ((Value & 0xffu) != ((Value>>8) & 0xffu) || Address_value)
	  {
	  // By convention we keep the high-byte value in the X register,
	  // but if we're using indexing we can't e.g. STX_IY because
	  // there is no such opcode.  Instead, we're forced to move
	  // the high-byte value through the A register as well.
	  // So instead of loading the high-byte it into the X register,
	  // we load the value into the A register and use that instead.
	  // Note we process the low-byte value through the A register too,
	  // but becaue we load/store load/store there is no conflict.
	  // The low-byte value has come and gone by the time we get there.
	  // [bj 14may2005]
	  //
	  switch (opcode_stx)
	    {
	    case ASM_STA_Y:
	    case ASM_STA_IY:		  // FIX WAS MISSING [bj 14may2005]
	      C.code.out(ASM_LDA_IMM);
	      break;

	    default:
	      C.code.out(ASM_LDX_IMM);
	    }


	  if (Address_value)
	    C.code.out_word(Value, upl_data_byte_hi);
	  else
	    C.code.out(Value>>8);

	  C.code.out(opcode_stx);
	  }
	else
	  C.code.out(opcode_sta);
      else
	{
	switch (Other_opcode)
	  {
	  case ASM_INC:
	  case ASM_DEC:
	    C.code.out(ASM_BNE);
	    C.code.out(zeropage ? 2 : 3);
	    break;

	  default:
	    if (opcode_stx == ASM_STA_Y)
	      C.code.out(ASM_TXA);
	    }

	  C.code.out(opcode_stx);
	}


      if (D.in_local_storage)
	C.code.out_word_patch(
		RUNTIME_DATASTACK, upl_patch_method_add, D.value+2);
      else if (zeropage)
	C.code.out(access_addr);
      else
	C.code.out_addr_data(access_addr+offset+1);

	  if (opcode_stx == ASM_LDA_IY) {	// HFD #2 fix for var = array[var]
	      C.code.out(ASM_TXA);
	      C.code.out(ASM_PLA);
	      }

// --------------------------------------------------------
// HFD this section unneeded - replaced by INY instructions
// --------------------------------------------------------

      /* UNSURE/COUT This code fragment increments CL... It belongs
		     somewhere above, but has lost its place!
		     See email of 14may2005 and above comment of same date.
		     [bj 21jun2005]

      // FIX: Was out of order; should be after we process address
      //      [bj 14may2005]
      //
      if (zeropage)
	{
        C.code.out(ASM_INC_Z);
	C.code.out(RUNTIME_CL);
	C.code.out(ASM_BNE);
	C.code.out(3);
	C.code.out(ASM_INC_Z);
	C.code.out(RUNTIME_CH);
	}
	*/
      }
      break;			    // OOPS! Break was too early! [bj 21jun2005]


    default:
      abend(WHERE0, "Bad case");
    }
  }


void upl_Compiler::asm_var(
		upl_Context&	C,
	const 	upl_Symbol& 	D,
		byte 		Offset,
		asm_opcode 	Opcode)
{
  C.code.out(Opcode);
  C.code.out_addr_data(D.value+Offset);
}




void upl_Compiler::store_transient(
		upl_Context&	  C,
	const	upl_Symbol& 	  Target,
	const 	upl_Expr_result&  Clause)
{
  C.rollback(Clause.transient_state);


  switch (Clause.transient)
    {
    case upl_transient_a:
      asm_var(C, Target, 0, ASM_STA);
      break;

    case upl_transient_ax:
      asm_var(C, Target, 0, ASM_STA);
      asm_var(C, Target, 1, ASM_STX);
      break;

    case upl_transient_x:
      asm_var(C, Target, 0, ASM_STX);
      break;

    case upl_transient_y:
      asm_var(C, Target, 0, ASM_STY);
      break;

    default:
      abend(WHERE0, "Bad case");
    }
}



void upl_Compiler::load_reg(
		upl_Context&	  C,
	const 	upl_Expr_result&  Clause)
{
  Select(Clause.is_transient() and Clause.transient != upl_transient_znc)
      C.rollback(Clause.transient_state);

      switch (Clause.transient)
	{
	case upl_transient_a:
	case upl_transient_ax:
	  break;

	case upl_transient_x:
	  C.code.out(ASM_TXA);
	  break;

	case upl_transient_y:
	  C.code.out(ASM_TYA);
	  break;

	case upl_transient_none:
	default:
	  abend(WHERE0, "Bad case");
	}


    when(Clause.is_variable() and Clause.variable->count == 0)
      reg_var(C, *Clause.variable, NULL, true, false);


    when(Clause.is_constant() and is_pointer(Clause.value_type))
      C.code.out(ASM_LDA);
      C.code.out_addr_data(Clause.value);
      C.code.out(ASM_LDX);
      C.code.out_addr_data(Clause.value+1);


    /*
    when(Clause.is_constant() and Clause.value_type == upl_pointer_byte)
      C.code.out(ASM_LDA_IMM);
      C.code.out_word((Clause.value), upl_data_byte_lo);
      C.code.out(ASM_LDX_IMM);
      C.code.out_word((Clause.value), upl_data_byte_hi);
      */


    when(Clause.is_constant())
      Select(0 <= Clause.value and Clause.value <= 255)
	  C.code.out(ASM_LDA_IMM);
	  C.code.out(Clause.value);

	otherwise
	  C.code.out(ASM_LDA_IMM);
	  C.code.out(Clause.value);
	  C.code.out(ASM_LDX_IMM);
	  C.code.out(Clause.value >> 8);
      endsel


    otherwise
      switch (Clause.value_bytes)
	{
	case 2:
	  C.code.out(ASM_JSR);
	  C.code.out_word_patch(RUNTIME_POP_W);
	  break;

	case 1:
	  C.code.out(ASM_JSR);
	  C.code.out_word_patch(RUNTIME_POP_B);
	  break;

	default:
	  abend(WHERE0, "Bad case");
	}
  endsel
}








upl_value_type upl_Compiler::op_result(
		upl_op		Op,
		upl_value_type	A,
		upl_value_type	B,
		upl_value_type&	new_A,
		upl_value_type&	new_B,
		boolean&	Unsigned)
{
  boolean		swapped = false;
#ifdef NO_WARNINGS
  upl_value_type	result = upl_void,
#else
  upl_value_type	result,
#endif
			temp;

  if (B < A)
    {
    swap(A, B, temp);
    swapped = true;
    }

  new_A 	= A,
  new_B 	= B;
  Unsigned	= false;


  Select(A == B)
      result = A;

    // A is byte or char, B is short or ushort.
    //
    when(A == upl_byte  or A == upl_char or A == upl_boolean)
      Select(B == upl_byte  or B == upl_char or B == upl_boolean)
	  result = B;

	when(B == upl_short or B == upl_ushort)
	  result = upl_short;
	  new_A = (A == upl_byte) ? upl_ushort : upl_short;
      endsel

    // A and B are Short and ushort
    //
    otherwise
      result = upl_ushort;
  endsel


  switch (Op)
    {
    case upl_gt:
    case upl_ge:
    case upl_lt:
    case upl_le:
    case upl_eq:
    case upl_ne:
      result = upl_boolean;
      break;

    case upl_mul:
    case upl_div:
    case upl_mod:
      if (result == upl_byte or result == upl_char)
	{
	result  =
	new_A   =
	new_B	= (result == upl_byte) ? upl_ushort : upl_short;
	}
#ifdef NO_WARNINGS
    default: ;
#endif
    }


//Unsigned = result == upl_ushort or result == upl_byte;
  Unsigned = new_A == upl_ushort or new_A == upl_byte;


  if (swapped)
    swap(new_A, new_B, temp);


  return result;
}



const upl_Symbol *upl_Compiler::type(
	Flex& 		L,
	upl_Context& 	C,
	boolean		Mandatory)
{
  upl_Symbol *S;


  S = (upl_Symbol *)C.symbols.get(L, L.peek(), false);


  if (S != NULL)
    {
    if (S->symbol_type != upl_type)
      return NULL;

    L.get_id(NULL);

    if (L.matches("*"))
      {
      upl_symbol_token	token;

      strcpy(token, S->token);
      strcat(token, "*");

      S = (upl_Symbol *)C.symbols.get(L, token, false);

      if (S == NULL)
	L.parse_error("In this implementation pointers of this type "
		      "are not currently supported");
      }
    }
  else
    Select(L.matches("struct"))
#ifndef NO_WARNINGS
	upl_value_type	   field_type;
#endif
	upl_Symbol 	  *D;
	upl_Symbol const  *T;
	long		   offset_i = 0;
	long		   field_i  = 0;


	S = C.symbols.declare(C.list_file, L, L.get_id(NULL), C.allocate_type(), upl_type, 0);

	L.check("{");

	while (not L.matches("}"))
	  {
	  T = type(L, C, true);

	  D = C.symbols.declare(C.list_file, L, L.get_id(NULL), T->value_type, upl_variable, offset_i);

	  offset_i += D->size_bytes;

	  S->fields.size(field_i+1);
	  S->fields.set(field_i, D);

	  L.check(";");
	  }

	S->size_bytes  =
	S->value_bytes = offset_i;


      when(L.matches("enum"))
	{
	upl_symbol_token  token;
	upl_symbol_token  type_token = "";
	long		  current_i = 0;
	upl_Expr_result	  clause;


	if (L.peek_type() == flex_id)
	  L.get_id(type_token);


	if (L.matches("{"))
	  {
	  loop
	    {
	    L.get_id(token);
	    if (L.matches("="))
	      expr(L, C, clause, 0);
	    else
	      clause.set_constant(L, current_i);

	    S = C.symbols.declare(C.list_file,
		  L, token, clause.value_type, upl_constant, clause.value);

	    current_i = clause.value+1;

	    if (not L.matches(","))
	      break;
	    }

	  L.check("}");
	  }


	// Don't use characters.
	//
	if (clause.value_type == upl_char)
	   clause.value_type = upl_byte;

	// If it's already declared then ignore it.
	//
	if ((S=(upl_Symbol *)C.symbols.get(L, type_token, false)) != NULL)
	  {
	  if (S->symbol_type != upl_type or
	      S->value_bytes == 0 or
	      S->value_bytes >  2)
	    L.parse_error("This type identifier has already been used.");
	  }
	else
	  {
	  if (type_token[0])
	    S = C.symbols.declare(C.list_file,
		  L, type_token,
		  clause.value_type,
		  upl_type, 0);
	  else
	    {
	    char const *T;

	    switch (clause.value_type)
	      {
	      case upl_byte: 		T = "byte"; 	break;
	      case upl_boolean: 	T = "boolean";	break;
	      case upl_char: 		T = "char"; 	break;
	      case upl_ushort: 		T = "ushort"; 	break;
	      case upl_short: 		T = "short"; 	break;
	      default:
#ifdef NO_WARNINGS
		T = "err";
#endif
		abend(WHERE0, "Bad case");
	      }


	    S = (upl_Symbol *)C.symbols.get(L, T, true);
	    }
	  }
	}

    otherwise
      S = NULL;
  endsel


  if (Mandatory)
    if (S == NULL)
      L.parse_error("Type expected");


  return S;
}




void upl_Compiler::append_mangling(
	char 	       Mangling[],
	upl_value_type Value_type)
{
  char  ch[3] = {0, 0, 0};
  int	chs=0;


  switch (Value_type)
    {
    case upl_pointer_byte:
    case upl_pointer_char:
    case upl_pointer_boolean:
    case upl_pointer_ushort:
    case upl_pointer_short:
      ch[chs++] = '*';
      switch (Value_type)
	{
	case upl_pointer_byte:	    Value_type = upl_byte;     break;
	case upl_pointer_char:      Value_type = upl_char;     break;
	case upl_pointer_boolean:   Value_type = upl_boolean;  break;
	case upl_pointer_ushort:    Value_type = upl_ushort;   break;
	case upl_pointer_short:     Value_type = upl_short;    break;
	default:;
	}
      break;

    default:;
    }



  switch (Value_type)
    {
    case upl_void:		ch[chs] = 'v'; break;
    case upl_byte:		ch[chs] = 'b'; break;
    case upl_char:		ch[chs] = 'c'; break;
    case upl_boolean:		ch[chs] = 'B'; break;
    case upl_ushort:		ch[chs] = 'u'; break;
    case upl_short:		ch[chs] = 's'; break;
    default:
      abend(WHERE0, "Bad case");
    }


  strcat(Mangling, ch);
}




char upl_Compiler::convert_char(
	char 			Ch,
	upl_char_conversion 	Conversion)
{
  switch (Conversion)
    {
    case upl_char_conversion_none:
	break;

    case upl_char_CBM_upper:
	// Screen: `A'=65.  No `a'.
	//
	Ch = toupper(Ch);
	break;

    case upl_char_CBM_lower:
	// Screen: `a' = 65, `A'=193.
	//
	Select('a' <= Ch and Ch <= 'z')
	    Ch = Ch - 'a' + 'A';

	  when('A' <= Ch and Ch <= 'Z')
	    Ch = Ch - 'A' + 193;

	  otherwise
	    ;
	endsel

    default:;
    }


  return Ch;
}




boolean upl_Compiler::is_pointer(upl_value_type Value_type)
{
  switch (Value_type)
    {
    case upl_pointer_byte:
    case upl_pointer_char:
    case upl_pointer_boolean:
    case upl_pointer_ushort:
    case upl_pointer_short:
      return true;

    default:
      return false;
    }
}




upl_value_type upl_Compiler::convert_pointer(Flex& L, upl_value_type Value_type)
{
  switch (Value_type)
    {
    case upl_byte:	return upl_pointer_byte;
    case upl_char:      return upl_pointer_char;
    case upl_boolean:   return upl_pointer_boolean;
    case upl_ushort:    return upl_pointer_ushort;
    case upl_short:     return upl_pointer_short;
    default:
      if (&L != NULL)
	L.parse_error("In this implementation no pointer may be defined for this type");
      else
	abend(WHERE0, "In this implementation no pointer may be defined for this type");
    }

  return upl_value_none;
}


upl_value_type upl_Compiler::pointer_value_type(upl_value_type Value_type)
{
  switch (Value_type)
    {
    case upl_pointer_byte:	return upl_byte;
    case upl_pointer_char:      return upl_char;
    case upl_pointer_boolean:   return upl_boolean;
    case upl_pointer_ushort:    return upl_ushort;
    case upl_pointer_short:     return upl_short;
    default:
      abend(WHERE0, "Bad case");
    }

  return upl_value_none;
}
