//
// NAME:
//   upl_link.cpp
// TITLE:
//   UPL/Quetzalcoatl: Linker.
// FUNCTION:
//   See header.
//
// AUTHOR:
//   Brendan Jones. (Contact through www.kdef.com/geek/vic)
// RIGHTS:
//   (c) Copyright Brendan Jones, 1998.  All Rights Reserved.
// SECURITY:
//   Unclassified.  
// LEGAL NOTICE:
//   See legal.txt before viewing, modifying or using this software.
// CONTACT:
//   Web:	http://www.kdef.com/geek/vic
//   Email:	See www.kdef.com/geek/vic
// DATE:
//   July 21, 1998.
// RIGHTS:
//  This file is part of The Quetzalcoatl Compiler.
//  
//  The Quetzalcoatl Compiler is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//  
//  The Quetzalcoatl Compiler is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//  
//  You should have received a copy of the GNU General Public License
//  along with The Quetzalcoatl Compiler; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
//
// MODIFICATIONS:
//   NAME  MOD  DATE     DESCRIPTION
//
//
#include "upl_link.h"
#include "implemen.h"

#ifdef NEW_CODE
  long		total_data	= 0;		// size of data segmnet
  upl_addr	data_1		= 0;		// offset of data segment
  upl_addr	Dataseg		= 0xffff;	// destination in RAM of data segment (from runtime module)
  char		machine		= 0;		// determines header to use
#endif


upl_Core::upl_Core(void)
{
  cores 	= 0;
  start_addr 	= 0;
}




void upl_Core::load(
	upl_addr Target,
	byte 	*Source,
	long	 Source_bytes)
{
  assert(0 <= Target-start_addr and
	 Target-start_addr+Source_bytes <= core.size());

  memcpy(&core[Target-start_addr], Source, Source_bytes);
}




long upl_Link::patch_externals(
	upl_Core&	Core,
	upl_addr	Base_addr,
	upl_Object&	Object,
	common_Keys&	Names,
	const char     *Object_name)
{
  long		 p;
  upl_Patch	*P;

#ifndef GCC
  union
    {
    ushort	*W;
    byte	*B;
    };
#else
    ushort T1;
    byte	*B = NULL;
#endif

#ifdef NO_WARNINGS
  char const	*name = NULL;
#else
  char const	*name;
#endif

  long		 external_addr;
  long		 errors = 0;
  long		 external_name_i;


  for (p=0; p<Object.patches; p++)
    {
    P = &Object.patch[p];

    if (P->external_id <= RUNTIME_END)
      name = runtime[P->external_id];
    else
      {
      external_name_i = P->external_id-(RUNTIME_END+1);

      if (0 <= external_name_i and
	  external_name_i < Object.external_names.size())
	name = Object.external_names[external_name_i].name;
      else
	abend(WHERE0, "External name index out of bounds");
      }

    if ((external_addr = Names.get(name)) >= 0)
      {
#ifndef GCC
       W = Core.physical_word(Base_addr + P->addr);
#endif
       switch (P->patch_method)
	 {
	 case upl_patch_method_none:
#ifdef GCC
	   Core.core[(long)(Base_addr + P->addr-Core.start_addr)] = (byte)(external_addr & 0xff);
	   Core.core[(long)(Base_addr + P->addr-Core.start_addr+1)] = (byte)(external_addr >> 8);
#else
	   *W  = external_addr;
#endif
	    break;

	 case upl_patch_method_add:
#ifdef GCC
	   T1 =
	    Core.core[(long)(Base_addr + P->addr-Core.start_addr)] +
	   (Core.core[(long)(Base_addr + P->addr-Core.start_addr+1)] << 8);
	    T1 += external_addr;
	    Core.core[(long)(Base_addr + P->addr-Core.start_addr)] = (byte)(T1 & 0xff);
	    Core.core[(long)(Base_addr + P->addr-Core.start_addr+1)] = (byte)(T1 >> 8);
#else
	    *W += external_addr;
#endif
	 break;

	 case upl_patch_method_set_high:
	    *B = (external_addr>>8);
	    break;

	 case upl_patch_method_set_low:
	    *B = (external_addr);
	    break;

	 default:
	   abend(WHERE0, "Bad case");
	 }
      }
    else
      {
      cerr << "E: Undefined external symbol " << name
	   << " found in module " << Object_name << ".\n";

      errors++;
      }
    }

  return errors;
}




void upl_Link::relocate(
	upl_Core&	Core,
	upl_addr	Reloc_addr,
	upl_Object&  	Map,
	upl_addr	Code_base_addr,
	upl_addr	Data_base_addr)
{
  long		 r;
  byte 		*B;
#ifndef GCC
  ushort	*W;
#else
  ushort T1;
#endif
  upl_reloc_type reloc_type;

#ifdef NO_WARNINGS
  upl_addr	 base_addr = 0;
#else
  upl_addr	 base_addr;
#endif


  long		 reloc_partner_i = 0;

  for (r=0; r<Map.objects; r++)
    if ((reloc_type=Map.reloc[r]) != upl_reloc_none)
      {
      switch (reloc_type bitand upl_reloc_segment)
	{

// need to determine if simple var or array/string
// if array/string then need to put in code segment
// and copy to data segment if changed

	case upl_reloc_data:
#ifdef NEW_CODE
	  if (data_1 == 0)		// nasty, but workable
	    data_1 = Data_base_addr;
	  if (Dataseg != 0xFFFF)
	    base_addr = Data_base_addr - data_1 + Dataseg;
	  else
	    base_addr = Data_base_addr;
#else
	  base_addr = Data_base_addr;
#endif
	  break;

	case upl_reloc_code:
	  base_addr = Code_base_addr;
	  break;

	default:
	  abend(WHERE0, "Bad case");
	}

      switch (reloc_type bitand upl_reloc_datatype)
	{
	case upl_reloc_word:
#ifndef GCC
	   W  = Core.physical_word(Reloc_addr+r);
	  *W  = (*W + base_addr);
#else
// get unrelocated address
	   T1 = 
	    Core.core[(long)(Reloc_addr+r-Core.start_addr)] +
	   (Core.core[(long)(Reloc_addr+r-Core.start_addr+1)] << 8);
// add appropriate relocation constant
	   T1 += base_addr;
// save it back to code
	   Core.core[(long)(Reloc_addr+r-Core.start_addr)] = (byte)(T1 & 0xff);
	   Core.core[(long)(Reloc_addr+r-Core.start_addr+1)] = (byte)(T1 >> 8);
#endif
// this fixes the references, but doesn't move the code/data to where it belongs
	   break;

	case upl_reloc_byte_lo:
	   B  = Core.physical_byte(Reloc_addr+r);
	  *B  = (*B + (Map.reloc_partner[reloc_partner_i++]<<8) + base_addr);
	   break;

	case upl_reloc_byte_hi:
	   B  = Core.physical_byte(Reloc_addr+r);
	  *B  = ((*B << 8) + Map.reloc_partner[reloc_partner_i++] + base_addr) >> 8;
	   break;

	default:
	  abend(WHERE0, "Bad case");
	}
      }

  assert(Map.reloc_partners == reloc_partner_i);
}





// This module used to combine one absolute (runtime library)
// code-only segment immediately followed by one relocatable
// data segment then one code segment
// now it puts the absolute and relocatable code sections
// together first
boolean upl_Link::link(
	upl_Core&		     	Core,
	Stingray_long<upl_Segments>& 	Seg,
	int				Verbose,
	upl_addr&			Entry_addr,
	upl_addr			Default_start_addr,
	long&				Datastack)
{
  common_Keys		names;
  upl_Segments	       *S;
  upl_Segments	       *SA 	= NULL; // Up to one absolute segment.
  upl_Name	       *N;
  int			seg_i;
  long			name_i;
  long			errors	 	= 0;

#ifndef NO_WARNINGS
  long			warnings 	= 0;
#endif

  long			core_bytes;
  upl_addr		reloc_addr;
  long			entry_addr;

  // Find the absolute segment.
  //
  core_bytes = 0;
  for (seg_i=0; seg_i<Seg.size(); seg_i++)
    if (Seg.occupied(seg_i))
      {
      S = &Seg[seg_i];

      core_bytes += S->code.objects + S->data.objects;

      if (S->absolute)
	if (SA == NULL)
	  SA = S;
	else
	  {
	  cerr << "E: Only one absolute segment is allowed" << endl;
	  errors++;
	  }
      }


  if (SA == NULL)
    {
    reloc_addr	    =
    Core.start_addr = Default_start_addr;

    if (Verbose >= 2)
      cerr << "I: No absolute segments found; "
	      "Will use default start address of "
	   << Core.start_addr << endl;
    }
  else
    {
    Core.start_addr = SA->code_addr;
    reloc_addr	    = SA->code_addr + SA->code.objects + SA->data.objects;
    }


  // Then the code segments.
  //
  for (seg_i=0; seg_i<Seg.size(); seg_i++)
    if (Seg.occupied(seg_i))
      {
      S = &Seg[seg_i];

      if (not S->absolute)
	{
	S->code_addr = reloc_addr;

	reloc_addr += S->code.objects;
	}
      }

  // Order the data segments last.
  //
  for (seg_i=0; seg_i<Seg.size(); seg_i++)
    if (Seg.occupied(seg_i))
      {
      S = &Seg[seg_i];

      if (not S->absolute)
	{
	S->data_addr = reloc_addr;

	reloc_addr += S->data.objects;
#ifdef NEW_CODE
	total_data += S->data.objects;
#endif
	}
      }


  // Load names.
  //
  for (seg_i=0; seg_i<Seg.size(); seg_i++)
    if (Seg.occupied(seg_i))
      {
      S = &Seg[seg_i];

      for (name_i=0; name_i<S->code.names.size(); name_i++)
	{
	N = &S->code.names[name_i];

	if (not names.insert(N->name,
	    S->absolute ? N->addr : N->addr+S->code_addr))
	  {
	  cerr << "E: " << S->segment_name
	       << " contains duplicate name " << N->name << endl;
	  errors++;
	  }
	}

      for (name_i=0; name_i<S->data.names.size(); name_i++)
	{
	N = &S->data.names[name_i];

	if (not names.insert(N->name,
	    S->absolute ? N->addr : N->addr+S->data_addr))
	  {
	  cerr << "E: " << S->segment_name
	       << " contains duplicate name " << N->name << endl;
	  errors++;
	  }
	}
      }


  if (errors)
    return false;


  Core.core.size(core_bytes);
  Core.cores = core_bytes;


  for (seg_i=0; seg_i<Seg.size(); seg_i++)
    if (Seg.occupied(seg_i))
      {
      S = &Seg[seg_i];


      if (S->absolute)
	{
	if (Verbose >= 2)
	  cerr << "I: Loading absolute segment " << S->segment_name
	       << " at " 	<< S->code_addr << endl;

	Core.load(S->code_addr, S->code.object.item, S->code.objects);
	}
      else
	{
	if (Verbose >= 2)
	  {
	  if (S->data.objects > 0)
	    cerr << "I: Loading "
		 << S->segment_name
		 << " data segment at "
		 << S->data_addr
		 << " (page alignment "
		 << (S->data_addr bitand 0xffu)
		 << " bytes)"
		 << endl;

	  if (S->code.objects > 0)
	    cerr << "I: Loading " << S->segment_name
		 << " code segment at "
		 << S->code_addr
		 << endl;
	  }

       if (not upl_Link::alignment_check(
	  S->segment_name, "data", S->data, S->data_addr))
         errors++;

       if (not upl_Link::alignment_check(
	  S->segment_name, "code", S->code, S->code_addr))
	 errors++;


	Core.load(S->code_addr,
		  S->code.object.item, S->code.objects);

	Core.load(S->data_addr,
		  S->data.object.item, S->data.objects);

#ifdef NEW_CODE
	Dataseg = names.get("dataseg");
	machine = names.get("machine");
#endif

	// relocate code references
	relocate(Core, S->code_addr, S->code,
		 S->code_addr, S->data_addr);

	// relocate data references
	//
	relocate(Core, S->data_addr, S->data,
		 S->code_addr, S->data_addr);

	errors += patch_externals(Core, S->code_addr, S->code, names, S->segment_name);
	errors += patch_externals(Core, S->data_addr, S->data, names, S->segment_name);

	cout.fill('0');
	if (Verbose >= 4)
	  {
#ifndef NEWCPP
	  cout << "I: Absolute code: " << hex;
#else
	  cout << "I: Absolute code: " << std::hex;
#endif

	  for (long i=0; i<S->code.objects; i++)
	    {
	    cout.width(2);
	    cout << (ushort)*Core.physical_byte(S->code_addr+i) << ' ';
	    }
	  cout << endl;
	  }
	}
      }

  Datastack = names.get("datastack");


  if ((entry_addr = names.get("main()")) < 0)
    entry_addr = names.get("main");


  if (entry_addr  >= 0)
    {
    if (errors == 0 and Verbose >= 2)
      cerr << "I: Entry point at address " << entry_addr << endl;

    Core.entry_addr = entry_addr;
    Entry_addr = entry_addr;
    }
  else
    {
    cerr << "E: main not declared; no entry point\n";
    errors++;
    }



  return errors == 0;
}




boolean upl_Core::save(
	const char *Exe_fn)
{

  FILE		*f;
  ushort	 load_addr = start_addr;
  assert(sizeof(load_addr) == 2);

#ifdef NEW_CODE
// AIM-65 header
if (machine == 9) {		// new header compatible with Daryl Rictor's 65C02 simulator
  byte newhead[10];
  ushort bsize;
  load_addr = start_addr - 6;
  bsize = cores + 6;

  newhead[0] = load_addr & 0xff;	// these 4 bytes are stripped off by the loader
  newhead[1] = load_addr >> 8;
  newhead[2] = bsize & 0xff;
  newhead[3] = bsize >> 8;

  newhead[4] = 0x20;			// place JSR to actual program start first
  newhead[5] = entry_addr & 0xff;
  newhead[6] = entry_addr >> 8;
  newhead[7] = 0x4C;			// JMP back to AIM-65 monitor
  newhead[8] = 0xA1;
  newhead[9] = 0xE1;

  file_open(f, Exe_fn, "wb");
  bwrite(&newhead[0],10, f)
}
// 7800 header
else if (machine == 10) {
  byte newhead[13];
  ushort bsize;
  load_addr = start_addr - 9;
  bsize = cores + 9;

  newhead[0] = load_addr & 0xff;	// these 4 bytes are stripped off by the datalinker
  newhead[1] = load_addr >> 8;
  newhead[2] = bsize & 0xff;
  newhead[3] = bsize >> 8;


  newhead[4] = 0x4C;			// place JSR to actual program start first
  newhead[5] = entry_addr & 0xff;
  newhead[6] = entry_addr >> 8;
  newhead[7] = data_1 & 0xff;		// where data is in file
  newhead[8] = data_1 >> 8;

  newhead[9] = Dataseg & 0xff;		// start of RAM data segment
  newhead[10] = Dataseg >> 8;

  newhead[11] = total_data & 0xff;	// size of data
  newhead[12] = total_data >> 8;

  file_open(f, Exe_fn, "wb");
  bwrite(&newhead[0],13, f)
}
// default/vic-20/c64 header
else {
  byte tmp;
  file_open(f, Exe_fn, "wb");
  tmp = load_addr & 0xff;
  bwrite1(&tmp, f);
  tmp = load_addr >> 8;
  bwrite1(&tmp, f);
}
#else
  file_open(f, Exe_fn, "wb");
  bwrite1(&load_addr, f);
#endif

#ifdef NEW_CODE
if (Dataseg != 0xffff)
	printf("I: Data offset: %04lx    size: %04lx    dest: %04lx\n", data_1-load_addr, total_data, Dataseg);
#endif

#ifdef NO_WARNINGS
  bwrite(core.item, (unsigned)cores, f);
#else
  bwrite(core.item, cores, f);
#endif

  file_close(f);

  return true;
}




boolean upl_Link::alignment_check(
	const char     *Object_name,
	const char     *Segment_name,
	upl_Object&	Object,
	long		Actual_addr)
{
  boolean	success = true;


  if (Object.absolute_aligned)
    if (Actual_addr != Object.alignment_addr)
      {
      success = false;

      cerr << "E: Misaligned Segment. The module "
	   << Object_name << ' ' << Segment_name
	   << " may only be loaded at absolute address "
	   <<  Object.alignment_addr
	   << ", but has been linked at "
	   <<  Actual_addr;
      }


  if (success)
    if (Object.page_aligned)

#ifdef NO_WARNINGS
      if ((Actual_addr & 0xffu) != (unsigned)Object.alignment_addr)
#else
      if ((Actual_addr & 0xffu) != Object.alignment_addr)
#endif

	{
	success = false;

	cerr << "E: Misaligned Segment. The module "
	     << Object_name << ' ' << Segment_name
	     << " may only be loaded at a page offset of "
	     <<  Object.alignment_addr
	     << ", but has been linked at a page offset of "
	     << (Actual_addr & 0xffu)
	     << ".  Review the -p options and recompile.\n";
	}


  /*
  if (not success)
    cerr << ".  In this configuration an executable cannot be generated.\n";
    */


  return success;
}










