//
// NAME:
//   upl_main.cpp
// TITLE:
//   UPL/Quetzalcoatl: Main (User Command Line Interface).
// FUNCTION:
//   See header.
//
// AUTHOR:
//   Brendan Jones. (Contact through www.kdef.com/geek/vic)
//   Bug extermination by Harry Dodgson
// RIGHTS:
//   (c) Copyright Brendan Jones, 1998-2005.  All Rights Reserved.
// SECURITY:
//   Unclassified.  
// LEGAL NOTICE:
//   See legal.txt before viewing, modifying or using this software.
// CONTACT:
//   Web:	http://www.kdef.com/geek/vic
//   Email:	See www.kdef.com/geek/vic
// DATE:
//   July 6, 1998.
// RIGHTS:
//  This file is part of The Quetzalcoatl Compiler.
//  
//  The Quetzalcoatl Compiler is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//   the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//  
//  The Quetzalcoatl Compiler is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//  
//  You should have received a copy of the GNU General Public License
//  along with The Quetzalcoatl Compiler; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//
//
// MODIFICATIONS:
//   NAME  MOD  DATE     DESCRIPTION
//
//

#ifdef DJGPP
#include <unistd.h>
#endif

#include "upl_link.h"
#include "implemen.h"
#include <sys/stat.h>
#if	defined(COMMON_DOS) || defined(COMMON_MSW)
#include <io.h>
#endif
#include "image/im_pcx.h"
#include <time.h>


#define UPL_MACRO_CHUNK	8


#ifdef	COMMON_DOS
#include <dos.h>

unsigned _stklen = 32768;
#endif


void usage(const char *Program = NULL,
	   const char *Option  = NULL,
	   const char *Message = NULL)
{
  if (Option)
    {
    if (Message)
      cerr << "Error: " << Message << " " << Option << endl;
    else
      cerr << "Error: " << Option << endl;

    cerr << "Type \""
	 << (Program == NULL ? "quetzalcoatl" : Program)
	 << " --help\" for help.\n";

    exit(8);
    }

  cerr <<
  "Usage: quetzalcoatl { options | -p bytes | -r target | -om outfile | file }\n"
  " -v      Verbose messages. (Repeat for more verbosity) (Default off).\n"
  " -l,-g   Link the executable (Default).   -g Beep on success (Default off).\n"
  " -o,-m   Name of -o executable file, -m next object file.\n"
  " -L,-c   Don't link the executable.  (-c also forces recompile.)\n"
  " -f,-F   Force/don't force (default) recompile. -C addr to load charset PCX.\n"
  " -r      Select runtime library apple, vicu, vicx, c64 or generic (Default).\n"
  "         vicu=unexpanded vic, vicx=expanded vic, generic=unexp vic or c64.\n"
  " -a      Assembly .lst file.  -A turns assembly .lst file off (Default).\n"
  " -s      List segment sizes (Default on).  -S Don't list segment sizes.\n"
  " -b      Create a BASIC bootstrap at start (Default), or -B in datastack.\n"
  " -R      List Runtime Symbol Table (for compiler debugging) (Default off).\n"
  " -T,-K   Convert strings/chars to Uppercase/Commodore lowercase. (Default off).\n"
  " -p      Set Data segment page alignment of next module. -P unknown (Default).\n"
  " -O opts Optimisation.  0-9=level. p=peephole. f=flow.  (Default off).\n"
  " -D     -Didentifier=integer defines a program constant.\n"
  " -x      Specify default start address for executable.  (Default 4096).\n"
#ifdef NEW_CODE
#endif
  " file    Input file extensions: c, cpp, c++, upl, asm, pcx/bin or obj.\n"
  "         The executable file may have any extension, but we recommend bin.\n";


  exit(8);
}


char const *title =
"Quetzalcoatl C/UPL 6502 Cross Compiler 2.1.0 BETA www.kdef.com/geek/vic";

char const *banner =
"Copyright Brendan Jones, 1998-2006. All Rights Reserved.\n"
"Debugged, augmented, and tested by Harry Dodgson, 2005.\n\n"
"You must read and agree to the terms in LEGAL.TXT before using this software.\n";


void bailout(void)
{
  cerr << "S: The compiler executable has been corrupted.\n"
	     "Recommend you reinstall the compiler.\n\b";
  exit(19);
}


const char *construct_fn(
	const char *Source_fn,
	      char *Object_fn,
	const char *Extension = "obj")
{
  char	*ext = NULL;
  char 	*p;



  strcpy(Object_fn, Source_fn);
  for (p=Object_fn; *p != 0; p++)
    switch (*p)
      {
      case '.': 		ext = &p[1]; break;
      case '/': case '\\':	ext = NULL;  break;
      default:;
      }


  if (ext != NULL)
    {
    strcpy(ext, Extension);
    return Object_fn;
    }
  else
    {
    usage(NULL, "Source files must have an extension; eg. .c, .asm");
    return NULL;
    }
}


const char *construct_fn_in_dir(
	      char *Buffer_fn,
	const char *Exe_fn,
	const char *Fn)
{
  char	*ext = Buffer_fn;
  char 	*p;



  strcpy(Buffer_fn, Exe_fn);
  for (p=Buffer_fn; *p != 0; p++)
    switch (*p)
      {
      case '/': case '\\':	ext = &p[1];  break;
      default:;
      }


  strcpy(ext, Fn);


  return Buffer_fn;
}


boolean	prefer_obj(
	const char *Source_fn,
	char 	   *Object_fn,
	char const *Overide_object_fn,
	const char *Extension = "obj")
{
  struct stat source_stat, object_stat;


  if (stat(Source_fn, &source_stat) != 0)
    {
    cerr << "E: Cannot open file " << Source_fn << endl;
    exit(10);
    }


  if (Overide_object_fn == NULL)
    construct_fn(Source_fn, Object_fn, Extension);
  else
    strcpy(Object_fn, Overide_object_fn);


  if (stat(Object_fn, &object_stat) != 0)
    return false;	// Only source is available;
  else
    //
    // Prefer obj if it was created after the source was last modified.
    //
    return object_stat.st_mtime > source_stat.st_mtime;
}



void load_macros(
	upl_Context&		  C,
	Mantaray_short<upl_Name>& Macro,
	short			  Macros)
{
  for (int name_i=0; name_i<Macros; name_i++)
    {
    Flex L;

    const upl_Name *N = &Macro[name_i];

    if (N->name[0])
      C.symbols.declare(NULL, L, N->name, upl_ushort, upl_constant, N->addr);
    }
}




void list_header(
	ostream 	&Str,
	const char 	*Source_fn,
	const char 	*Object_fn,
	const char 	*List_fn)
{
  time_t timer;
  struct tm *tblock;

  /* gets time of day */
  timer = time(NULL);

  /* converts date/time to a structure */
  tblock = localtime(&timer);


  Str <<
  ";\n"
  ";  FILE:     " << List_fn   << endl <<
  ";\n"
  ";  SOURCE:   " << Source_fn << endl <<
  ";  OBJECT:   " << Object_fn << endl <<
  ";  DATE:     Compiled: " << asctime(tblock)   <<
  ";  COMPILER: " << title << endl <<
  ";\n"
  ";\n";
}




long roms_read  = 0;
const long roms = 2048;
byte rom[2048];




void compile_PCX(
	const char *PCX_fn,
	const char *Bin_fn,
	long	    Start_addr,
	boolean	    Double_height)
{

#ifdef NO_WARNINGS
  ushort 	rom_i, col_i, row_i, char_i;
#else
  ushort 	rom_i, col, col_i, row_i, char_i;
#endif

  byte 	       *rom_char;
  const long    indent = 8;
  ushort	height = Double_height ? 16 : 8;


  {
  im_PCX I((char *)PCX_fn);

  for (rom_i=0; rom_i<roms; rom_i++)
    rom[rom_i] = 0;

  for (int line=0; line<indent+128; line++)
    {
    I.next_line();

    if (line >= indent)
      {
      row_i = (line-indent) % height;

      for (int col=0; col<128; col++)
	if (I.line[indent+col] != 0)
	  {
	  col_i  = col%8;
	  char_i = ((line-indent)/height * 16) + (col/8);

	  rom_char = &rom[char_i*8UL+row_i];

	  *rom_char = *rom_char bitor (0x80 >> col_i);
	  }
      }
    }
  }


  FILE *f;
  ushort addr = Start_addr >= 0 ? Start_addr : 0x1400 ;


  file_open(f, Bin_fn, "wb");

  bwrite1(&addr, f);

  bwrite(rom, sizeof(rom), f);

  file_close(f);

  roms_read = sizeof(rom);
}





int main(int argc, const char *argv[])
{
  Flex	 	 L;
  upl_Context   *C;

#ifdef NO_WARNINGS
  upl_Segments 	*S = NULL;
#else
  upl_Segments 	*S;
  boolean	compiled	= false;
  char const   *PCX_fn			= NULL;
#endif

  char const	*ext;
  char const	*fn;
  int		 fn_length;
  upl_Assembler *A = NULL;
  int		 verbose	= 0;
  int		 segs 		= 0;
  Stingray_long<upl_Segments>	seg;
  int 		argi;
  char		object_fn[max_filename];
  char		exe_fn[max_filename]	= {0};
  char		rtime_fn[max_filename]	= {0};
  char		list_fn[max_filename];
  boolean	link_exe	= true;
  boolean	needs_runtime	= false;
  boolean	loaded_runtime	= false;
  boolean	success = false;
  boolean	force_recompile = false;
  boolean	list_assembler	= false;
  char 		runtime_fn[max_filename] = "uplrtime.obj";
  char		target[16] 	      = "generic";	// generic
  boolean	segment_sizes	= true;
  boolean	checksum_error	= true;
  boolean	boot_stub	= false;
  boolean	boot_start	= true;
  boolean	list_runtable	= false;
  long		source_files	= 0;
  boolean	bell		= false;
  upl_char_conversion char_conversion   = upl_char_conversion_none;
  upl_char_conversion string_conversion = upl_char_conversion_none;
  long		page_align_bytes	= -1;
  long		default_start_addr = 0x1000;
  int		optimise_level		= 0;
  boolean	optimise_peephole	= false;
  boolean	optimise_flow		= false;
  char const   *module_fn		= NULL;
  long		charset_rom_addr	= -1;

  Mantaray_short<upl_Name>	macro;
  short		macros 	= 0;
  ofstream     *list	= NULL;



  cerr << '\n' << title << '\n' << banner;
  cerr
  << "This program is distributed in the hope that it will be useful,\n"
     "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
     "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
     "GNU General Public License for more details.\n\n";


  if (argc == 1 or argv[1][0] == '?' or equal(argv[1], "--help"))
    usage(argv[0], NULL);


  cerr
  << "WARNING: This is a BETA release, built on "
  << __DATE__ << " at " << __TIME__ << ".\n"
  << "         This BETA release is incomplete and known to contain bugs.\n\n";



  segs = 1;
  for (argi=1; argi<argc; argi++)
    if (argv[argi][0] != '-')
      segs++;
  seg.size(segs);
  segs = 0;

//HFD once should be enough
//  if (checksum_error)
//    bailout();


  for (argi=1; argi<argc; argi++)
    if (argv[argi][0] == '-')
      for (char const *p=&argv[argi][1]; *p != 0; p++)
	switch (*p)
	  {
	  case 'v': verbose++; break;
	  case 'V': verbose = 0; break;
	  case 'l': link_exe = true; break;
	  case 'L': link_exe = false; break;
	  case 'c': link_exe = false; force_recompile = true;   break;
	  case 'f': force_recompile = true;   break;
	  case 'F': force_recompile = false;  break;
	  case 'a': list_assembler  = true;   break;
	  case 'A': list_assembler  = false;  break;
	  case 's': segment_sizes   = true;   break;
	  case 'S': segment_sizes   = false;  break;
	  case 'b': boot_stub       = true;   boot_start = true; break;
	  case 'B': boot_stub       = false;  boot_start = false; break;
	  case 'R': list_runtable   = true;   break;
	  case 'g': bell	    = true;   break;
	  case 'k': char_conversion = upl_char_CBM_lower; break;
	  case 'K': char_conversion = upl_char_CBM_upper; break;
	  case 't': string_conversion = upl_char_CBM_lower; break;
	  case 'T': string_conversion = upl_char_CBM_upper; break;
	  case 'P': page_align_bytes  = -1; break;

	  case 'D':
	    {
	    upl_Name temp;
	    upl_Name *N = &temp;

#ifdef NO_WARNINGS
	    unsigned int name_i = 0;
#else
	    int name_i = 0;
#endif

	    for (++p; *p != '=' and *p != 0; p++)
	      if (name_i < sizeof(N->name))
		N->name[name_i++] = *p;

	    if (name_i < sizeof(N->name))
	      N->name[name_i++] = 0;
	    else
	      N->name[sizeof(N->name)-1] = 0;


	    if (*p == '=')
	      if (isdigit(p[1]))
		N->addr = atol(&p[1]);
	      else
		usage(argv[0], "-D value must be 0..65535");
	    else
	      N->addr = 0;


	    N = NULL;
	    for (int macro_i=0; macro_i<macros; macro_i++)
	      if (strncmp(temp.name, macro[macro_i].name, sizeof(temp.name)) == 0)
		{
		N = &macro[macro_i];
		break;
		}


	    if (N != NULL)
	      N->addr = temp.addr;
	    else
	      {
	      if (macros >= macro.size())
		macro.size(macro.size()+UPL_MACRO_CHUNK);

	      macro[macros++] = temp;
	      }


	    // HACK: so it looks like we're at the end of this option.
	    //
	    if (*p == 0)
	      p--;
	    else
	      for (; p[1] != 0; p++)
		;


	    break;
	    }


	  case 'o':
	    if (argi+1 <= argc)
	      strcpy(exe_fn, argv[++argi]);
	    else
	      usage(argv[0], "-o must be followed by executable filename.");
	    break;


	  case 'C':
	    if (argi+1 <= argc and isdigit(argv[argi+1][0]))
	      charset_rom_addr = atol(argv[++argi]);
	    else
	      usage(argv[0], "-C must be followed by the charset rom address.");
	    break;


	  case 'm':
	    if (argi+1 <= argc)
	      {
	      module_fn = argv[++argi];
	      int module_len = strlen(module_fn);

	      if (module_len < 4 or
		  notequal(&module_fn[module_len-4], ".obj"))
		usage(argv[0], "-m object filename must have a .obj extension.");
	      }
	    else
	      usage(argv[0], "-m must be followed by an object filename.");
	    break;


	  case 'O':
	    if (argi+1 <= argc)
	      {
	      for (char const *p=argv[++argi]; *p != 0; p++)
		if (isdigit(*p))
		  optimise_level = *p - '0';
		else switch (*p)
		  {
		  case 'p': optimise_peephole = true; break;
		  case 'f': optimise_flow     = true; break;
		  default:;
		  }
	      }
	    else
	      usage(argv[0], "-O must be followed by optimisation switches.");
	    break;


	  case 'x':
	    if (argi+1 <= argc and isdigit(argv[argi+1][0]))
	      {
	      default_start_addr = atol(argv[++argi]);
	      }
	    else
	      usage(argv[0], "-x must be followed by default executable start address");
	    break;


	  case 'p':
	    if (argi+1 <= argc and isdigit(argv[argi+1][0]))
	      {
	      page_align_bytes = atol(argv[++argi]);
	      }
	    else
	      usage(argv[0],
		    "-p must be followed by the page alignment offset "
		    "in bytes.");
	    break;


	  case 'r':
	    if (argi+1 <= argc)
	      {
	      if (equal(argv[argi+1], "generic"))
		{
		strcpy(runtime_fn, "uplrtime.obj");
		argi++;
		}
	      else
		{
		strcpy(runtime_fn, "uplr");
		strncat(runtime_fn, argv[++argi], 4);
		strcat(runtime_fn, ".obj");
		}
	       strcpy(target, argv[argi]);

	      if (verbose >= 1)
		cerr << "I: -r option selects runtime library "
		     << runtime_fn << " (Target " << target << ").\n";
	      }
	    else
	      usage(argv[0], "-r must be followed by the runtime target.");
	    break;

	  default:
	    usage(argv[0], argv[argi-1], "Unrecognised option");
	  }
    else
      {
      fn_length = strlen(fn=argv[argi]);

      if (fn_length >= 4)
	{
	ext = &argv[argi][fn_length-4];

	Select(equal(ext, ".upl") or
	       equal(&argv[argi][fn_length-2], ".c")   or
	       equal(ext, ".c++") or
	       equal(ext, ".cpp"))

	    needs_runtime = true;
	    source_files++;

	    if (exe_fn[0] == 0)
	      construct_fn(fn, exe_fn, "bin");


	    if (prefer_obj(fn, object_fn, module_fn) and
		not force_recompile)
	      {
	      if (verbose >= 1)
		cerr << "I: Using object file " << object_fn << endl;

	      mcheck(S = new upl_Segments);
	      S->read(object_fn);
	      }
	    else
	      {
	      if (verbose >= 1)
		cerr << "I: Compiling file "
		     << fn << "\t-> " << object_fn;

	      mcheck(S = C = new upl_Context);
	      S->absolute 	    = false;
	      C->verbose 	    = verbose;
	      C->char_conversion    = char_conversion;
	      C->string_conversion  = string_conversion;


	      C->optimise_level	    = optimise_level;
	      C->optimise_peephole  = optimise_peephole;
	      C->optimise_flow	    = optimise_flow;


	      if (list_assembler)
		{
		construct_fn(fn, list_fn, "lst");

		mcheck(list = new ofstream(list_fn));

		list_header(*list, fn, object_fn, list_fn);

		*list << "\t\t  .data\n";

		C->list_file = list;
		}


	      if (page_align_bytes >= 0)
		{
		if (verbose >= 2)
		  cerr << " (data page alignment "
		       << page_align_bytes
		       << " bytes).";

		C->know_alignment   = true;
		C->page_align_bytes = page_align_bytes;
		}


	      if (verbose >= 1)
		cerr << endl;


	      load_macros(*C, macro, macros);

	      L.open_file((char *)argv[argi],
		  flex_syntax_c|flex_syntax_no_real|flex_syntax_lit_char);
	      upl_Compiler::program(L, *C, true);
	      L.close();

	      {
	      char const *p;
	      char const *root = argv[argi];

	      for (p=argv[argi]; *p != 0; p++)
		if (*p == '/' or *p == '\\')
		  root = p;

	      int count = 0;
	      for (p=root; *p != 0 and *p != '.'; p++)
		count++;

	      strncpy(S->segment_name, root, count);
	      S->segment_name[count] = 0;
	      }


	      if (list)
		{
		*list << "\t\t  .code\n";

		*list << S->code;
		list->close();

		list = NULL;
		}



	      S->write(object_fn);


	      if (verbose >= 4)
		{
#ifndef NEWCPP
		cout << "I: Relocatable Code segment:" << hex;
#else
		cout << "I: Relocatable Code segment:" << std::hex;
#endif
		if (verbose >= 5)
		  cout << endl;

		for (long i=0; i<C->code.objects; i++)
		  {
		  if (verbose >= 5)
		    cout << i << ":\t" ;

		  cout << (ushort)C->code.object[i] << ' ';
		  }

		if (verbose == 4)
		  cout << endl;

		cout << endl;
		}

	      if (verbose >= 3)
		{
		upl_Patch const *P, *Q = NULL;

		cout << "I: Relocatable Code patches:";
		for (long i=0; i< C->code.patches; i++)
		  {
		  P = &C->code.patch[i];

		  if (i == 0
		   or P->external_id  != Q->external_id
		   or P->patch_method != Q->patch_method)
		     cout << ' ' << *P <<  '.';
		  else
#ifndef NEWCPP
		    cout << ' ' << hex << P->addr << dec << '.';
#else
		    cout << ' ' << std::hex << P->addr << std::dec << '.';
#endif

		  Q = P;
		  }
		cout << endl;

		cout << "I: Relocatable Exported names:";
#ifndef GCC
		for (i=0; i< C->code.names.size(); i++)
#else
		for (long i=0; i< C->code.names.size(); i++)
#endif
#ifndef NEWCPP
		  cout << ' ' << C->code.names[i].name
		       << " is at "
		       << hex << C->code.names[i].addr << dec << '.';
		cout << endl;
#else
		  cout << ' ' << C->code.names[i].name
		       << " is at "
		       << std::hex << C->code.names[i].addr << std::dec << '.';
		cout << endl;
#endif
		}
	      }



	  when(equal(ext, ".asm"))
	    boolean is_runtime = strncmp(fn, "uplr", 4) == 0;

	    source_files++;

	    if (exe_fn[0] == 0 and not is_runtime)
	      construct_fn(fn, exe_fn, "bin");

	    if (is_runtime)
	      loaded_runtime = true;

	    if (prefer_obj(fn, object_fn, module_fn) and
		not force_recompile)
	      {
	      if (verbose >= 1)
		cerr << "I: Using object file " << object_fn << endl;

	      mcheck(S = new upl_Segments);
	      S->read(object_fn);
	      }
	    else
	      {
	      if (verbose >= 1)
		cerr << "I: Compiling file "
		     << fn << "\t-> " << object_fn << endl;

	      if (A == NULL)
		mcheck(A = new upl_Assembler);

	      mcheck(S = C = new upl_Context);
	      S->absolute = true;
	      C->verbose = verbose;

	      load_macros(*C, macro, macros);

	      if (list_assembler)
		{
		construct_fn(fn, list_fn, "lst");

		mcheck(list = new ofstream(list_fn));

		list_header(*list, fn, object_fn, list_fn);

#ifndef NEWCPP
		*list << hex;
#else
		*list << std::hex;
#endif

		list->fill('0');

		C->list_file = list;
		}


	      A->program(argv[argi], *C, true);



	      if (list)
		{
		list->close();

		list = NULL;
		}


	      {
	      char const *p;
	      char const *root = argv[argi];

	      for (p=argv[argi]; *p != 0; p++)
		if (*p == '/' or *p == '\\')
		  root = p;

	      int count = 0;
	      for (p=root; *p != 0 and *p != '.'; p++)
		count++;

	      strncpy(S->segment_name, root, count);
	      S->segment_name[count] = 0;
	      }

	      S->write(object_fn);
	      }


	  when(equal(ext, ".obj"))
	      if (verbose >= 1)
		cerr << "I: Using object file " << object_fn << endl;

	      mcheck(S = new upl_Segments);
	      S->read(fn);


	  when(equal(ext, ".pcx"))
	    if (roms_read > 0)
	      usage(argv[0], "Only one .pcx file may be included");

	    if (prefer_obj(fn, object_fn, module_fn, "bin") and
		not force_recompile)
	      {
	      if (verbose >= 1)
		cerr << "I: Using character set binary file "
		     << object_fn << endl;

	      ushort addr;
	      FILE *f;

	      file_open(f, object_fn, "rb");

	      bread1(&addr, f);
#ifdef FIXEND
	      wordswap(addr);
#endif
	      roms_read = fread(rom, sizeof(char), roms, f);

	      file_close(f);

	      if (roms_read <= 0)
		abend(WHERE0, "Corrupted character set binary file", 0, object_fn);
	      }
	    else
	      {
	      if (verbose >= 1)
		cerr << "I: Compiling file "
		     << fn << "\t-> " << object_fn << endl;


	      compile_PCX(fn, object_fn, charset_rom_addr, false);
	      }

	    S = NULL;


	  otherwise
	    usage(argv[argi], "Unrecognised file extension");
	endsel


	if (S != NULL)
	  {
	  seg.set(segs++, S);

	  page_align_bytes = -1;
	  module_fn 	   = NULL;
	  }
	}
      else
	usage(argv[argi], "Files must have an extension");
      }


  boolean target_commodore =
	equal(target, "vicu") 	or
	equal(target, "vicx") 	or
	equal(target, "vicg") 	or
	equal(target, "c64")  	or
	equal(target, "generic");



  if (needs_runtime and not loaded_runtime)
    {
    char const *rtime_obj = NULL;


    if (access(rtime_obj=runtime_fn, 0) != 0)
      {
      construct_fn_in_dir(rtime_fn, argv[0], runtime_fn);

      if (access(rtime_obj=rtime_fn, 0) != 0)
	{
	strcpy(rtime_fn, "/usr/local/lib/");
	strcat(rtime_fn, runtime_fn);

	if (access(rtime_obj=rtime_fn, 0) != 0)
	  rtime_obj = NULL;
	}
      }


    if (rtime_obj != NULL)
      {
      if (verbose >= 1)
	cerr << "I: Automatically including runtime library "
	     << rtime_obj << ".\n";

      mcheck(S = new upl_Segments);
      S->read(rtime_obj);

      seg.set(segs++, S);

      success = true;

      loaded_runtime = true;
      }
    else
      {
      cerr << "E: Could not find runtime library "
	   << runtime_fn << ".  Unable to link executable.\n";

      if (equal(target, "vicu") or equal(target, "vicx") or
	  equal(target, "c64"))
	cerr <<
	"I: I noticed you used the -r option to specify a Commodore target.  "
	"Might I suggest you use the default runtime library which "
	"works with both the VIC-20 and C64?  Simply omit the -r option.\n";

      success = false;
      }
    }
  else
    success = true;



  if (A)
    delete A;


  if (segment_sizes)
    for (int seg_i=0; seg_i<segs; seg_i++)
      if (seg.occupied(seg_i))
	{
	S = &seg[seg_i];

	cerr << "I: Module " << S->segment_name
	     << ".\tSegment Sizes: "
	     << "Code: ";

	cerr.width(6);
	cerr << S->code.objects << " bytes. "
	     << "Data: ";
	cerr.width(6);
	cerr << S->data.objects << " bytes.\n";
	}


  char const *boot_message = "";

  if (success)
    if (link_exe and source_files > 0)
      {
      success = false;
      long  datastack = -1;

      upl_Core	core;
      upl_addr 	entry;


      if (exe_fn[0] == 0)
	strcpy(exe_fn, "aout.bin");


      //assert(seg.size() == segs);
      if (upl_Link::link(core, seg, verbose, entry, default_start_addr, datastack))
	{
	if (roms_read > 0)
	  {
	  if (charset_rom_addr < 0)
	    charset_rom_addr = core.start_addr;

	  if (charset_rom_addr < core.start_addr)
	    cerr << "E: Character set may not be linked before the "
		    "executable load address " << core.start_addr
		 << '.' << endl;
	  else
	    {
	    boolean prune = false;

	    if (equal(target, "vicu"))
	      if (core.cores < 3584)
		prune = true;


	    if (charset_rom_addr+roms_read-core.start_addr > core.core.size())
	      {
	      core.cores = charset_rom_addr+roms_read-core.start_addr;

	      core.core.size(core.cores);
	      }

	    if (prune and core.cores > 3584)
	      {
	      core.cores = 3584;


	      if (verbose >= 2)
		cerr << "I: Pruning character set to fit inside "
			"available memory.\n";
	      }


	    memcpy(&core.core[charset_rom_addr-core.start_addr], rom, roms_read);


	    if (verbose >= 2)
	      cerr << "I: Loading character set data segment at "
		   << charset_rom_addr
		   << ".\n";


	      if (segment_sizes)
		{
		cerr << "I: Module character set."
		     << " Segment Size:                     Data: ";
		cerr.width(6);
		cerr << roms_read << " bytes.\n";
		}

	    success = true;
	    }
	  }
	else
	  success = true;


	if (success and boot_stub)
	  {
	  if (core.cores >= 16 and target_commodore)
	    {
	    byte *boot;

	    if (boot_start)
	      boot =  core.core.item;
	    else if (datastack >= 0)
	      boot = &core.core[datastack - core.start_addr];
	    else
	      {
	      boot = NULL;
	      cerr << "E: datastack not defined, "
			 "so no bootstrap was created.";
	      }

	    if (boot)
	      {
	      for (int boot_i=0; boot_i<16; boot_i++)
		boot[boot_i] = 0;

	      boot[1] = (core.start_addr+12);
	      boot[2] = (core.start_addr+12)>>8;
	      boot[3] = 0x66;
	      boot[4] = 0x19;
	      boot[5] = 158;	// CBM SYS

	      #ifdef COMMON_MSVC
	      _ultoa(entry, (char *)&boot[6], 10);
	      #elif defined(DJGPP)
	      sprintf((char *)&boot[6], "%ld", (long)entry);
	      #elif defined(COMMON_DOS)
	      ultoa(entry, &boot[6], 10);
	      #elif defined(GCC)
	      sprintf((char *)&boot[6], "%ld", (long)entry);
	      #else
	      sprintf(&boot[6], "%ld", (long)entry);
	      #endif

	      cerr << "I: Created a one-time BASIC Bootstrap ("
		      "assuming BASIC begins at "
		   << core.start_addr << ").\n";

	      if (charset_rom_addr == core.start_addr)
		cerr << "W: The first 16 bytes of the character set "
			"have been overwritten by the bootstrap. "
			"The first two characters will be corrupted.\n";

	      /*
	      if (not loaded_runtime)
		cerr << "W: You're not using a standard runtime library; "
			"The BASIC Bootstap has overwritten the first "
			"16 bytes of whatever assembly module it is that "
			"you're using.\n";
	       */

	      boot_message = " or RUN";
	      }
	    }
	  else
	    cerr << "E: I don't know how to create a "
		    "BASIC Bootstrap for this target.\n";
	  }


	if (core.start_addr + core.cores > 0xffff)
	  cerr <<
	  "E: Executable is too large. "
	  "It would start at address " << core.start_addr <<
	  " and be " << core.cores << " bytes long. "
	  "This would exceed the 6502 64Kb memory addressing limit by " <<
	  (core.cores - core.start_addr) << " bytes.";


	else if (core.save(exe_fn))
	  {
	  success = true;
	  cerr << "I: Executable starts at address " << core.start_addr
	       << ", and is " << core.cores << " bytes long.\n";
	  cerr << "I: Success!  Saved executable " << exe_fn << ".  ";
	  cerr << "Entry point at " << entry << ".\n";


	  if (verbose == 0)
	    Select(target_commodore)
		if (equal(target, "vicg"))
		  cerr << "I: This is a vicg executable; "
		  "You must reconfigure memory before loading it:\n"
		  "   poke 36869,224:poke648,24:print chr$(147) "
		  "or use a vicg snapshot.\n"
		   "I: To then run the executable on an expanded VIC-20 "
		   "then type SYS " << entry << boot_message << ".\n";
		else
		  cerr << "I: To run the executable on a VIC-20/C64 "
			  "load it and type SYS " << entry
		       << boot_message << ".\n";

		if (equal(target, "vicu") or equal(target, "generic"))
		  if (core.cores > 3584)
		    {
		    cerr <<
		    "W: This is too large to run on an unexpanded VIC.";

		    if (equal(target, "vicu"))
		      cerr << "  Try relinking with -r vicx instead.";

		    cerr << endl;
		    }


	      when(equal(target, "apple"))
		cerr << "I: To run the executable on an Apple ][ "
		      "load it and type CALL " << entry
		     << boot_message << ".\n";
	    endsel
	  }
	}
      }


  if (list_runtable)
    {
    int	runtable_i;

    cerr << "I: Runtime Symbol Table:\n";

    cerr.width(8);
    cerr << "#" << "  Symbol Name\n";

    for (runtable_i=0; runtime[runtable_i] != NULL; runtable_i++)
      {
      cout.width(8);
      cout << runtable_i << ": " << runtime[runtable_i] << endl;
      }
    }



  if (not success)
    cerr << "E: Errors were found, and so no executable was created.\n";
  else if (not link_exe and verbose >= 1)
    cerr << "I: Not linking, so no executable was created.\n";

  if (bell)
    {
    cerr << '\a';

    /*
    if (not success)
      cerr << '\a';
    */
    }


  return success ? 0 : 4;
}

